<?php
/**
 * Plugin Name: SMSTunnel
 * Plugin URI: https://smstunnel.io/plugins/smstunnel
 * Description: SMS gateway using your own phone. Provides API integration, settings, and SMS-based Two-Factor Authentication for WordPress login.
 * Version: 1.0.1
 * Author: SMSTunnel.io
 * Author URI: https://smstunnel.io
 * Text Domain: smstunnel
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @author Narcis Bodea <contact@narboweb.ro>
 * @copyright 2024-2025 NARBOWEB SRL
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SMSTUNNEL_VERSION', '1.0.1');
define('SMSTUNNEL_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMSTUNNEL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SMSTUNNEL_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('SMSTUNNEL_API_BASE', 'https://smstunnel.io/api/v1');


/**
 * Sanitize roles array
 */
function smstunnel_sanitize_roles($input) {
    if (!is_array($input)) {
        return array();
    }
    return array_map('sanitize_text_field', $input);
}

/**
 * Main SMSTunnel Plugin Class
 */
class SMSTunnel {

    /**
     * Single instance of the class
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
    }

    /**
     * Initialize the plugin
     */
    public function init() {
        // Load custom language if set, otherwise use site language
        $custom_lang = get_option('smstunnel_language', '');
        if (!empty($custom_lang)) {
            add_filter('plugin_locale', function($locale, $domain) use ($custom_lang) {
                if ($domain === 'smstunnel') {
                    return $custom_lang;
                }
                return $locale;
            }, 10, 2);
        }

        // Text domain loaded automatically by WordPress since 4.6

        // Include required files
        $this->includes();

        // Ensure database table exists
        $this->maybe_create_tables();

        // Admin menu and settings
        add_action('admin_menu', array($this, 'add_admin_menu'), 9);
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_init', array($this, 'handle_oauth_callback'));
        add_action('admin_init', array($this, 'handle_disconnect'));

        // Admin scripts
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // AJAX handlers
        add_action('wp_ajax_smstunnel_send_sms', array($this, 'ajax_send_sms'));
        add_action('wp_ajax_smstunnel_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_smstunnel_social_login', array($this, 'ajax_social_login'));
        add_action('wp_ajax_smstunnel_email_login', array($this, 'ajax_email_login'));
        add_action('wp_ajax_smstunnel_verify_api_key', array($this, 'ajax_verify_api_key'));
        add_action('wp_ajax_smstunnel_e2e_start', array($this, 'ajax_e2e_start'));
        add_action('wp_ajax_smstunnel_e2e_status', array($this, 'ajax_e2e_status'));
        add_action('wp_ajax_smstunnel_e2e_remove', array($this, 'ajax_e2e_remove'));

        // WordPress Login 2FA
        if (get_option('smstunnel_wp_2fa_enabled', 'no') === 'yes') {
            add_filter('authenticate', array($this, 'wp_2fa_authenticate'), 30, 3);
            add_action('login_form', array($this, 'wp_2fa_login_form'));
            add_action('wp_ajax_nopriv_smstunnel_wp_send_2fa', array($this, 'ajax_wp_send_2fa'));
            add_action('wp_ajax_nopriv_smstunnel_verify_phone_setup', array($this, 'ajax_verify_phone_setup'));
            add_action('wp_ajax_nopriv_smstunnel_confirm_phone_setup', array($this, 'ajax_confirm_phone_setup'));
            add_action('wp_ajax_nopriv_smstunnel_skip_phone_setup', array($this, 'ajax_skip_phone_setup'));
        }

        // Add phone field to user profile
        add_action('show_user_profile', array($this, 'add_phone_field_to_profile'));
        add_action('edit_user_profile', array($this, 'add_phone_field_to_profile'));
        add_action('personal_options_update', array($this, 'save_phone_field'));
        add_action('edit_user_profile_update', array($this, 'save_phone_field'));

        // Initialize REST API
        SMSTunnel_REST_API::get_instance();
        SMSTunnel_Quick_Setup::get_instance();
    }

    /**
     * Include required files
     */
    private function includes() {
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-api.php';
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-encryption.php';
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-rest-api.php';
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-quick-setup.php';
    }

    /**
     * Create database tables if they don't exist
     */
    private function maybe_create_tables() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'smstunnel_log';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) !== $table_name) {
            $charset_collate = $wpdb->get_charset_collate();

            $sql = "CREATE TABLE $table_name (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                phone varchar(50) NOT NULL,
                message text NOT NULL,
                status varchar(20) NOT NULL DEFAULT 'pending',
                message_id varchar(100) DEFAULT NULL,
                error_message text DEFAULT NULL,
                is_2fa tinyint(1) NOT NULL DEFAULT 0,
                order_id bigint(20) DEFAULT NULL,
                source varchar(50) DEFAULT 'manual',
                created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY phone (phone),
                KEY status (status),
                KEY created_at (created_at)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }

    /**
     * Add standalone admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('SMSTunnel', 'smstunnel'),
            __('SMSTunnel', 'smstunnel'),
            'manage_options',
            'smstunnel',
            array($this, 'render_settings_page'),
            'dashicons-smartphone',
            30
        );

        add_submenu_page(
            'smstunnel',
            __('Settings', 'smstunnel'),
            __('Settings', 'smstunnel'),
            'manage_options',
            'smstunnel',
            array($this, 'render_settings_page')
        );

        add_submenu_page(
            'smstunnel',
            __('Send SMS', 'smstunnel'),
            __('Send SMS', 'smstunnel'),
            'manage_options',
            'smstunnel-send',
            array($this, 'render_send_sms_page')
        );

        add_submenu_page(
            'smstunnel',
            __('SMS History', 'smstunnel'),
            __('SMS History', 'smstunnel'),
            'manage_options',
            'smstunnel-history',
            array($this, 'render_sms_history_page')
        );

        add_submenu_page(
            'smstunnel',
            __('Inbox', 'smstunnel'),
            __('Inbox', 'smstunnel'),
            'manage_options',
            'smstunnel-inbox',
            array($this, 'render_inbox_page')
        );

        add_submenu_page(
            'smstunnel',
            __('Developer Docs', 'smstunnel'),
            __('Developer Docs', 'smstunnel'),
            'manage_options',
            'smstunnel-docs',
            array($this, 'render_docs_page')
        );
    }

    /**
     * Register standalone settings
     */
    public function register_settings() {
        // API Settings
        register_setting('smstunnel_settings', 'smstunnel_server_url', array('sanitize_callback' => 'esc_url_raw'));
        register_setting('smstunnel_settings', 'smstunnel_api_key', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('smstunnel_settings', 'smstunnel_device_id', array('sanitize_callback' => 'sanitize_text_field'));

        // Language Settings
        register_setting('smstunnel_settings', 'smstunnel_language', array('sanitize_callback' => 'sanitize_text_field'));

        // 2FA Settings
        register_setting('smstunnel_settings', 'smstunnel_wp_2fa_enabled', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_roles', array('sanitize_callback' => 'smstunnel_sanitize_roles'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_expiry', array('sanitize_callback' => 'absint'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_length', array('sanitize_callback' => 'absint'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_message', array('sanitize_callback' => 'sanitize_textarea_field'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_server_fallback', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_remember_device', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('smstunnel_settings', 'smstunnel_2fa_remember_days', array('sanitize_callback' => 'absint'));

        // E2E Encryption Settings
        register_setting('smstunnel_settings', 'smstunnel_device_public_key', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('smstunnel_settings', 'smstunnel_e2e_device_id', array('sanitize_callback' => 'sanitize_text_field'));

        // API Section
        add_settings_section(
            'smstunnel_api_section',
            __('API Configuration', 'smstunnel'),
            array($this, 'api_section_callback'),
            'smstunnel_settings'
        );

        add_settings_field(
            'smstunnel_server_url',
            __('Server URL', 'smstunnel'),
            array($this, 'server_url_field_callback'),
            'smstunnel_settings',
            'smstunnel_api_section'
        );

        add_settings_field(
            'smstunnel_api_key',
            __('API Key', 'smstunnel'),
            array($this, 'api_key_field_callback'),
            'smstunnel_settings',
            'smstunnel_api_section'
        );

        add_settings_field(
            'smstunnel_device_id',
            __('Device ID', 'smstunnel'),
            array($this, 'device_id_field_callback'),
            'smstunnel_settings',
            'smstunnel_api_section'
        );

        // General Settings Section
        add_settings_section(
            'smstunnel_general_section',
            __('General Settings', 'smstunnel'),
            array($this, 'general_section_callback'),
            'smstunnel_settings'
        );

        add_settings_field(
            'smstunnel_language',
            __('Language', 'smstunnel'),
            array($this, 'language_field_callback'),
            'smstunnel_settings',
            'smstunnel_general_section'
        );

        // 2FA Section
        add_settings_section(
            'smstunnel_2fa_section',
            __('Two-Factor Authentication (2FA)', 'smstunnel'),
            array($this, 'twofa_section_callback'),
            'smstunnel_settings'
        );

        add_settings_field(
            'smstunnel_wp_2fa_enabled',
            __('WordPress Login 2FA', 'smstunnel'),
            array($this, 'wp_2fa_enabled_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_roles',
            __('Require 2FA for roles', 'smstunnel'),
            array($this, 'twofa_roles_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_expiry',
            __('Code expiry (minutes)', 'smstunnel'),
            array($this, 'twofa_expiry_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_length',
            __('Code length', 'smstunnel'),
            array($this, 'twofa_length_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_message',
            __('2FA Message Template', 'smstunnel'),
            array($this, 'twofa_message_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_server_fallback',
            __('Server Fallback', 'smstunnel'),
            array($this, 'twofa_fallback_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_remember_device',
            __('Remember Device', 'smstunnel'),
            array($this, 'twofa_remember_device_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        add_settings_field(
            'smstunnel_2fa_remember_days',
            __('Remember Duration (days)', 'smstunnel'),
            array($this, 'twofa_remember_days_field_callback'),
            'smstunnel_settings',
            'smstunnel_2fa_section'
        );

        // E2E Encryption Section
        add_settings_section(
            'smstunnel_e2e_section',
            __('End-to-End Encryption', 'smstunnel'),
            array($this, 'e2e_section_callback'),
            'smstunnel_settings'
        );

        add_settings_field(
            'smstunnel_device_public_key',
            __('Device Public Key', 'smstunnel'),
            array($this, 'e2e_public_key_field_callback'),
            'smstunnel_settings',
            'smstunnel_e2e_section'
        );
    }

    /**
     * API section description
     */
    public function api_section_callback() {
        echo '<p>' . wp_kses(
            sprintf(
                /* translators: %s: link to dashboard */
                __('Configure your SMSTunnel API credentials. Get your API key from %s', 'smstunnel'),
                '<a href="https://smstunnel.io/dashboard/api-keys" target="_blank">smstunnel.io/dashboard</a>'
            ),
            array('a' => array('href' => array(), 'target' => array()))
        ) . '</p>';
    }

    /**
     * Server URL field
     */
    public function server_url_field_callback() {
        $value = get_option('smstunnel_server_url', 'https://smstunnel.io/api/v1');
        echo '<input type="text" name="smstunnel_server_url" value="' . esc_attr($value) . '" class="regular-text" placeholder="https://smstunnel.io/api/v1">';
        echo '<p class="description">' . esc_html__('API endpoint URL: https://smstunnel.io/api/v1', 'smstunnel') . '</p>';
    }

    /**
     * API Key field - Shows connection options
     */
    public function api_key_field_callback() {
        $value = get_option('smstunnel_api_key', '');
        $user = get_option('smstunnel_user', array());
        $quick_setup_url = admin_url('admin.php?page=smstunnel-quick-setup');
        $is_connected = !empty($value) && !empty($user) && is_array($user) && !empty($user['email']);

        // Verify API key is still valid on server
        $key_valid = false;
        $key_error = '';
        if ($is_connected) {
            $api = new SMSTunnel_API();
            $result = $api->test_connection();
            $key_valid = $result['success'];
            if (!$key_valid) {
                $key_error = $result['error'] ?? __('API key is invalid or expired', 'smstunnel');
            }
        }

        // Show connected state
        if ($is_connected && $key_valid) {
            echo '<div class="smstunnel-connected" style="max-width: 400px;">';

            // Connected user info
            echo '<div class="smstunnel-connected-user" style="padding: 20px; background: #e7f5e7; border: 1px solid #46b450; border-radius: 8px; margin-bottom: 15px;">';
            echo '<div style="display: flex; align-items: center; gap: 12px; margin-bottom: 12px;">';
            echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="#46b450"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>';
            echo '<div>';
            echo '<strong style="font-size: 14px; color: #2e7d32;">' . esc_html__('Connected', 'smstunnel') . '</strong><br>';
            $name = trim(($user['firstName'] ?? '') . ' ' . ($user['lastName'] ?? ''));
            if (empty($name)) {
                $name = $user['email'];
            }
            echo '<span style="color: #555;">' . esc_html($name) . '</span>';
            if (!empty($user['firstName']) || !empty($user['lastName'])) {
                echo ' <span style="color: #888;">(' . esc_html($user['email']) . ')</span>';
            }
            echo '</div>';
            echo '</div>';

            // Show API key (masked)
            echo '<div style="background: #f5f5f5; padding: 10px; border-radius: 4px; font-family: monospace; font-size: 13px; color: #666;">';
            echo esc_html__('API Key:', 'smstunnel') . ' ' . esc_html(substr($value, 0, 8) . '...' . substr($value, -4));
            echo '</div>';
            echo '</div>';

            // Disconnect button
            echo '<a href="' . esc_url(admin_url('admin.php?page=smstunnel&smstunnel_disconnect=1&_wpnonce=' . wp_create_nonce('smstunnel_disconnect'))) . '" class="button" style="margin-top: 10px; color: #a00;" onclick="return confirm(\'' . esc_js(__('Are you sure you want to disconnect?', 'smstunnel')) . '\');">';
            echo esc_html__('Disconnect', 'smstunnel');
            echo '</a>';

            // Hidden field to preserve API key
            echo '<input type="hidden" name="smstunnel_api_key" value="' . esc_attr($value) . '">';

            echo '</div>';
            return;
        }

        // Show invalid/expired API key error
        if ($is_connected && !$key_valid) {
            echo '<div class="smstunnel-connection-error" style="max-width: 400px;">';

            // Error message
            echo '<div style="padding: 20px; background: #fef0f0; border: 1px solid #d63638; border-radius: 8px; margin-bottom: 15px;">';
            echo '<div style="display: flex; align-items: flex-start; gap: 12px;">';
            echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="#d63638" style="flex-shrink: 0;"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>';
            echo '<div>';
            echo '<strong style="font-size: 14px; color: #d63638;">' . esc_html__('Connection Error', 'smstunnel') . '</strong><br>';
            echo '<span style="color: #555;">' . esc_html($key_error) . '</span>';
            echo '</div>';
            echo '</div>';

            // Show stored user info
            $name = trim(($user['firstName'] ?? '') . ' ' . ($user['lastName'] ?? ''));
            if (empty($name)) {
                $name = $user['email'] ?? '';
            }
            if (!empty($name)) {
                echo '<div style="margin-top: 10px; color: #666; font-size: 13px;">';
                echo esc_html__('Previously connected as:', 'smstunnel') . ' <strong>' . esc_html($name) . '</strong>';
                echo '</div>';
            }
            echo '</div>';

            // Disconnect button to clear and reconnect
            echo '<a href="' . esc_url(admin_url('admin.php?page=smstunnel&smstunnel_disconnect=1&_wpnonce=' . wp_create_nonce('smstunnel_disconnect'))) . '" class="button button-primary" style="margin-top: 10px;">';
            echo esc_html__('Reconnect', 'smstunnel');
            echo '</a>';

            echo '</div>';
            return;
        }

        // Connection options container - stacked layout (only show when not connected)
        echo '<div class="smstunnel-connection-options" style="display: flex; flex-direction: column; gap: 15px; margin-top: 10px; max-width: 400px;">';

        // Option 1: API Key (first)
        echo '<div class="smstunnel-option" style="padding: 20px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 8px; text-align: center;">';
        echo '<h4 style="margin: 0 0 15px 0; font-size: 14px; color: #333;">' . esc_html__('Connect with API Key', 'smstunnel') . '</h4>';
        echo '<input type="password" id="smstunnel-api-key-input" value="" class="regular-text" placeholder="sk_live_..." style="width: 100%; margin-bottom: 8px;">';
        echo '<input type="hidden" name="smstunnel_api_key" id="smstunnel-api-key-hidden" value="' . esc_attr($value) . '">';
        echo '<button type="button" id="smstunnel-api-key-connect" class="button button-primary" style="width: 100%; margin-bottom: 8px;" disabled>' . esc_html__('Connect', 'smstunnel') . '</button>';
        echo '<div id="smstunnel-api-key-status" style="margin-bottom: 8px;"></div>';
        echo '<p class="description" style="margin: 0; font-size: 12px;">' . esc_html__('Get your API key from smstunnel.io', 'smstunnel') . '</p>';
        echo '</div>';

        // Option 2: Social Login (Google/Facebook)
        echo '<div class="smstunnel-option" style="padding: 20px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 8px; text-align: center;">';
        echo '<h4 style="margin: 0 0 15px 0; font-size: 14px; color: #333;">' . esc_html__('Connect with', 'smstunnel') . '</h4>';
        echo '<div style="display: flex; flex-direction: column; gap: 10px;">';

        // Google button
        echo '<button type="button" id="smstunnel-google-login" class="button" style="display: inline-flex; align-items: center; justify-content: center; gap: 8px; padding: 8px 16px; width: 100%;">';
        echo '<svg width="18" height="18" viewBox="0 0 24 24"><path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/><path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/><path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/><path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/></svg>';
        echo esc_html__('Google', 'smstunnel');
        echo '</button>';

        // Facebook button
        echo '<button type="button" id="smstunnel-facebook-login" class="button" style="display: inline-flex; align-items: center; justify-content: center; gap: 8px; padding: 8px 16px; width: 100%; background: #1877F2; border-color: #1877F2; color: white;">';
        echo '<svg width="18" height="18" viewBox="0 0 24 24" fill="white"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>';
        echo esc_html__('Facebook', 'smstunnel');
        echo '</button>';

        echo '</div>';
        echo '</div>';

        // Option 3: QR Code (Quick Setup)
        echo '<div class="smstunnel-option" style="padding: 20px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 8px; text-align: center;">';
        echo '<h4 style="margin: 0 0 15px 0; font-size: 14px; color: #333;">' . esc_html__('Scan QR Code', 'smstunnel') . '</h4>';
        echo '<a href="' . esc_url($quick_setup_url) . '" class="button button-primary" style="display: inline-flex; align-items: center; justify-content: center; gap: 8px; padding: 8px 16px; width: 100%;">';
        echo '<svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor"><path d="M3 11h8V3H3v8zm2-6h4v4H5V5zM3 21h8v-8H3v8zm2-6h4v4H5v-4zM13 3v8h8V3h-8zm6 6h-4V5h4v4zM13 13h2v2h-2zM15 15h2v2h-2zM13 17h2v2h-2zM17 13h2v2h-2zM19 15h2v2h-2zM17 17h2v2h-2zM15 19h2v2h-2zM19 19h2v2h-2z"/></svg>';
        echo esc_html__('Quick Setup', 'smstunnel');
        echo '</a>';
        echo '<p class="description" style="margin: 8px 0 0 0; font-size: 12px;">' . esc_html__('Scan with SMSTunnel app', 'smstunnel') . '</p>';
        echo '</div>';

        // Option 4: Email/Password Login (last)
        echo '<div class="smstunnel-option" style="padding: 20px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 8px;">';
        echo '<h4 style="margin: 0 0 15px 0; font-size: 14px; color: #333; text-align: center;">' . esc_html__('Connect with Email', 'smstunnel') . '</h4>';
        echo '<div id="smstunnel-email-login-form" style="display: flex; flex-direction: column; gap: 10px;">';
        echo '<input type="email" id="smstunnel-login-email" placeholder="' . esc_attr__('Email', 'smstunnel') . '" class="regular-text" style="width: 100%;">';
        echo '<input type="password" id="smstunnel-login-password" placeholder="' . esc_attr__('Password', 'smstunnel') . '" class="regular-text" style="width: 100%;">';
        echo '<button type="button" id="smstunnel-email-login" class="button button-primary" style="width: 100%;">' . esc_html__('Login', 'smstunnel') . '</button>';
        echo '<p class="description" style="margin: 5px 0 0 0; font-size: 12px; text-align: center;">';
        /* translators: %s: registration URL */
echo wp_kses(sprintf(__('Don\'t have an account? <a href="%s" target="_blank">Register</a>', 'smstunnel'), 'https://smstunnel.io/register'), array('a' => array('href' => array(), 'target' => array())));
        echo '</p>';
        echo '</div>';
        echo '</div>';

        echo '</div>'; // End connection options

        // Social login notice container
        echo '<div class="smstunnel-social-login" style="display: none;"></div>';
    }

    /**
     * Device ID field
     */
    public function device_id_field_callback() {
        $value = get_option('smstunnel_device_id', '');
        echo '<input type="text" name="smstunnel_device_id" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">' . esc_html__('Optional. Specify a device ID to use for sending SMS. Leave empty to use any available device.', 'smstunnel') . '</p>';
    }

    /**
     * General section description
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__('General plugin settings.', 'smstunnel') . '</p>';
    }

    /**
     * Language field
     */
    public function language_field_callback() {
        $site_locale = get_locale();
        $saved_value = get_option('smstunnel_language', '');

        $languages = array(
            /* translators: %s: site locale code */
            ''      => sprintf(__('Site default (%s)', 'smstunnel'), $site_locale),
            'en_US' => 'English',
            'ro_RO' => 'Română',
        );

        echo '<select name="smstunnel_language" id="smstunnel_language">';
        foreach ($languages as $code => $name) {
            $selected = ($saved_value === $code) ? 'selected' : '';
            echo '<option value="' . esc_attr($code) . '" ' . esc_attr($selected) . '>' . esc_html($name) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select the language for SMS messages. Default uses the site language.', 'smstunnel') . '</p>';
    }

    /**
     * 2FA section description
     */
    public function twofa_section_callback() {
        echo '<p>' . esc_html__('Add SMS-based two-factor authentication. Users need a phone number in their profile.', 'smstunnel') . '</p>';
    }

    /**
     * WordPress 2FA enabled field
     */
    public function wp_2fa_enabled_field_callback() {
        $value = get_option('smstunnel_wp_2fa_enabled', 'no');
        echo '<label><input type="checkbox" name="smstunnel_wp_2fa_enabled" value="yes" ' . checked($value, 'yes', false) . '> ';
        echo esc_html__('Enable SMS 2FA for WordPress login (wp-login.php)', 'smstunnel') . '</label>';
    }

    /**
     * 2FA Server Fallback field
     */
    public function twofa_fallback_field_callback() {
        $value = get_option('smstunnel_2fa_server_fallback', 'yes');
        echo '<label><input type="checkbox" name="smstunnel_2fa_server_fallback" value="yes" ' . checked($value, 'yes', false) . '> ';
        echo esc_html__('Allow login without 2FA when SMS server is unavailable', 'smstunnel') . '</label>';
        echo '<p class="description" style="color: #d63638;">' . esc_html__('Security note: When enabled, users can login without 2FA if the SMS server cannot be reached. This prevents lockouts but reduces security.', 'smstunnel') . '</p>';
    }

    /**
     * 2FA Remember Device field
     */
    public function twofa_remember_device_field_callback() {
        $value = get_option('smstunnel_2fa_remember_device', 'no');
        echo '<label><input type="checkbox" name="smstunnel_2fa_remember_device" value="yes" ' . checked($value, 'yes', false) . '> ';
        echo esc_html__('Allow users to remember this device', 'smstunnel') . '</label>';
        echo '<p class="description">' . esc_html__('Users can skip 2FA on trusted devices. Device trust is based on browser and IP address.', 'smstunnel') . '</p>';
    }

    /**
     * 2FA Remember Days field
     */
    public function twofa_remember_days_field_callback() {
        $value = get_option('smstunnel_2fa_remember_days', 30);
        echo '<input type="number" name="smstunnel_2fa_remember_days" value="' . esc_attr($value) . '" min="1" max="365" style="width:80px;"> ';
        echo esc_html__('days', 'smstunnel');
        echo '<p class="description">' . esc_html__('How long a device stays trusted before requiring 2FA again.', 'smstunnel') . '</p>';
    }

    /**
     * 2FA roles field
     */
    public function twofa_roles_field_callback() {
        $selected_roles = get_option('smstunnel_2fa_roles', array('administrator'));
        if (!is_array($selected_roles)) {
            $selected_roles = array('administrator');
        }
        $roles = wp_roles()->get_names();
        foreach ($roles as $role_key => $role_name) {
            $checked = in_array($role_key, $selected_roles) ? 'checked' : '';
            echo '<label style="display:inline-block; margin-right:15px; margin-bottom:5px;"><input type="checkbox" name="smstunnel_2fa_roles[]" value="' . esc_attr($role_key) . '" ' . esc_attr($checked) . '> ';
            echo esc_html($role_name) . '</label>';
        }
        echo '<p class="description">' . esc_html__('Select which user roles require 2FA.', 'smstunnel') . '</p>';
    }

    /**
     * 2FA expiry field
     */
    public function twofa_expiry_field_callback() {
        $value = get_option('smstunnel_2fa_expiry', 5);
        echo '<input type="number" name="smstunnel_2fa_expiry" value="' . esc_attr($value) . '" min="1" max="30" style="width:80px;"> ';
        echo esc_html__('minutes', 'smstunnel');
    }

    /**
     * 2FA code length field
     */
    public function twofa_length_field_callback() {
        $value = get_option('smstunnel_2fa_length', 6);
        echo '<select name="smstunnel_2fa_length">';
        for ($i = 4; $i <= 8; $i++) {
            $selected = ($value == $i) ? 'selected' : '';
            echo '<option value="' . intval($i) . '" ' . esc_attr($selected) . '>' . intval($i) . ' ' . esc_html__('digits', 'smstunnel') . '</option>';
        }
        echo '</select>';
    }

    /**
     * 2FA message template field
     */
    public function twofa_message_field_callback() {
        $default = __('Your {site_name} verification code is: {code}. Valid for {expiry} minutes.', 'smstunnel');
        $value = get_option('smstunnel_2fa_message', $default);
        echo '<textarea name="smstunnel_2fa_message" rows="3" style="width:100%; max-width:500px;">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">' . esc_html__('Placeholders: {code}, {site_name}, {expiry}', 'smstunnel') . '</p>';
    }

    /**
     * E2E Encryption section description
     */
    public function e2e_section_callback() {
        echo '<p>' . esc_html__('Enable end-to-end encryption so only your phone can read SMS messages. The server cannot decrypt them.', 'smstunnel') . '</p>';
    }

    /**
     * E2E Public Key field with QR pairing
     */
    public function e2e_public_key_field_callback() {
        // Single device E2E - get stored key
        $e2e_key = get_option('smstunnel_e2e_key', array());

        if (!empty($e2e_key) && !empty($e2e_key['public_key'])) {
            // Show paired device info
            $fingerprint = substr(hash('sha256', $e2e_key['public_key']), 0, 16);
            $device_name = $e2e_key['device_name'] ?: __('Phone', 'smstunnel');
            $paired_date = !empty($e2e_key['paired_at']) ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($e2e_key['paired_at'])) : '-';

            echo '<div style="padding: 15px; background: #e7f5e7; border: 1px solid #46b450; border-radius: 8px; max-width: 450px;">';
            echo '<div style="display: flex; align-items: center; gap: 10px; margin-bottom: 12px;">';
            echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="#46b450"><path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm-6 9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2zm3.1-9H8.9V6c0-1.71 1.39-3.1 3.1-3.1 1.71 0 3.1 1.39 3.1 3.1v2z"/></svg>';
            echo '<strong style="color: #2e7d32; font-size: 15px;">' . esc_html__('E2E Encryption Active', 'smstunnel') . '</strong>';
            echo '</div>';
            echo '<table style="width: 100%; font-size: 13px;">';
            echo '<tr><td style="color: #666; padding: 4px 0;">' . esc_html__('Device:', 'smstunnel') . '</td><td style="padding: 4px 0;"><strong>' . esc_html($device_name) . '</strong></td></tr>';
            echo '<tr><td style="color: #666; padding: 4px 0;">' . esc_html__('Fingerprint:', 'smstunnel') . '</td><td style="padding: 4px 0;"><code style="font-size: 11px; background: #f5f5f5; padding: 2px 6px; border-radius: 3px;">' . esc_html($fingerprint) . '</code></td></tr>';
            echo '<tr><td style="color: #666; padding: 4px 0;">' . esc_html__('Paired:', 'smstunnel') . '</td><td style="padding: 4px 0;">' . esc_html($paired_date) . '</td></tr>';
            echo '</table>';
            echo '<p style="margin: 12px 0 0 0; font-size: 12px; color: #666;">' . esc_html__('All SMS messages are encrypted before leaving WordPress.', 'smstunnel') . '</p>';
            echo '<button type="button" id="smstunnel-remove-e2e" class="button" style="margin-top: 12px; color: #a00;">' . esc_html__('Remove E2E Pairing', 'smstunnel') . '</button>';
            echo '</div>';
        } else {
            // Show QR pairing button
            echo '<div id="smstunnel-e2e-container" style="max-width: 450px;">';
            echo '<p style="margin-bottom: 15px; color: #666;">' . esc_html__('Pair your phone to enable end-to-end encryption. Messages will be encrypted before leaving WordPress.', 'smstunnel') . '</p>';
            echo '<button type="button" id="smstunnel-start-e2e" class="button button-primary" style="display: flex; align-items: center; gap: 8px;">';
            echo '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm-6 9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2zm3.1-9H8.9V6c0-1.71 1.39-3.1 3.1-3.1 1.71 0 3.1 1.39 3.1 3.1v2z"/></svg>';
            echo esc_html__('Start E2E Pairing', 'smstunnel');
            echo '</button>';
            echo '<div id="smstunnel-e2e-qr" style="margin-top: 15px; display: none;"></div>';
            echo '<div id="smstunnel-e2e-status" style="margin-top: 10px;"></div>';
            echo '</div>';
        }

        // Define smstunnel_admin inline to ensure it's available
        echo '<script>
            window.smstunnel_admin = window.smstunnel_admin || {
                ajaxUrl: "' . esc_js(admin_url('admin-ajax.php')) . '",
                nonce: "' . esc_js(wp_create_nonce('smstunnel_nonce')) . '"
            };
        </script>';
    }

    /**
     * Render standalone settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form action="options.php" method="post">
                <?php
                settings_fields('smstunnel_settings');
                do_settings_sections('smstunnel_settings');
                ?>

                <h2><?php esc_html_e('Test Connection', 'smstunnel'); ?></h2>
                <p>
                    <button type="button" id="smstunnel-test-connection" class="button button-secondary">
                        <?php esc_html_e('Test API Connection', 'smstunnel'); ?>
                    </button>
                    <span id="smstunnel-test-result" style="margin-left: 10px;"></span>
                </p>

                <?php submit_button(__('Save Settings', 'smstunnel')); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render Send SMS page
     */
    public function render_send_sms_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Send SMS', 'smstunnel'); ?></h1>

            <div class="card" style="max-width: 600px; padding: 20px;">
                <h2><?php esc_html_e('Send a Test SMS', 'smstunnel'); ?></h2>

                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="sms-phone"><?php esc_html_e('Phone Number', 'smstunnel'); ?></label></th>
                        <td>
                            <input type="text" id="sms-phone" class="regular-text" placeholder="+40712345678">
                            <p class="description"><?php esc_html_e('Use E.164 format (e.g., +40712345678)', 'smstunnel'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sms-message"><?php esc_html_e('Message', 'smstunnel'); ?></label></th>
                        <td>
                            <textarea id="sms-message" class="large-text" rows="4" placeholder="<?php esc_html_e('Enter your message...', 'smstunnel'); ?>"></textarea>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="button" id="smstunnel-send-test-sms" class="button button-primary">
                        <?php esc_html_e('Send SMS', 'smstunnel'); ?>
                    </button>
                    <span id="smstunnel-send-result" style="margin-left: 10px;"></span>
                </p>
            </div>
        </div>

        <script>
        jQuery(function($) {
            $('#smstunnel-send-test-sms').on('click', function() {
                var $btn = $(this);
                var $result = $('#smstunnel-send-result');
                var phone = $('#sms-phone').val().trim();
                var message = $('#sms-message').val().trim();

                if (!phone || !message) {
                    $result.html('<span style="color:red;"><?php esc_html_e('Phone and message are required.', 'smstunnel'); ?></span>');
                    return;
                }

                $btn.prop('disabled', true);
                $result.html('<span style="color:blue;"><?php esc_html_e('Sending...', 'smstunnel'); ?></span>');

                $.post(ajaxurl, {
                    action: 'smstunnel_send_sms',
                    nonce: '<?php echo esc_attr(wp_create_nonce('smstunnel_nonce')); ?>',
                    phone: phone,
                    message: message
                }, function(response) {
                    $btn.prop('disabled', false);
                    if (response.success) {
                        $result.html('<span style="color:green;">' + response.data.message + '</span>');
                        $('#sms-message').val('');
                    } else {
                        $result.html('<span style="color:red;">' + (response.data ? response.data.message : 'Error') + '</span>');
                    }
                }).fail(function() {
                    $btn.prop('disabled', false);
                    $result.html('<span style="color:red;"><?php esc_html_e('Request failed.', 'smstunnel'); ?></span>');
                });
            });
        });
        </script>
        <?php
    }

    /**
     * Render SMS History page
     */
    public function render_sms_history_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'smstunnel_log';

        // Pagination
        $per_page = 25;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Viewing data only
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;

        // Filter by status
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Viewing data only
        $status_filter = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '';
        $where = '';
        if (!empty($status_filter)) {
            $where = $wpdb->prepare(' WHERE status = %s', $status_filter);
        }

        // Get total count
        if (empty($status_filter)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
            $total_items = $wpdb->get_var("SELECT COUNT(*) FROM `{$wpdb->prefix}smstunnel_log`");
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
            $total_items = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `{$wpdb->prefix}smstunnel_log` WHERE status = %s", $status_filter));
        }
        $total_pages = ceil($total_items / $per_page);

        // Get SMS log
        if (empty($status_filter)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
            $sms_log = $wpdb->get_results($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}smstunnel_log` ORDER BY created_at DESC LIMIT %d, %d", $offset, $per_page));
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
            $sms_log = $wpdb->get_results($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}smstunnel_log` WHERE status = %s ORDER BY created_at DESC LIMIT %d, %d", $status_filter, $offset, $per_page));
        }

        // Status counts
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table aggregation
        $status_counts = $wpdb->get_results(
            "SELECT status, COUNT(*) as count FROM `{$wpdb->prefix}smstunnel_log` GROUP BY status",
            OBJECT_K
        );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('SMS History', 'smstunnel'); ?></h1>

            <ul class="subsubsub">
                <li>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-history')); ?>" <?php echo empty($status_filter) ? 'class="current"' : ''; ?>>
                        <?php esc_html_e('All', 'smstunnel'); ?>
                        <span class="count">(<?php echo intval($total_items); ?>)</span>
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-history&status=sent')); ?>" <?php echo $status_filter === 'sent' ? 'class="current"' : ''; ?>>
                        <?php esc_html_e('Sent', 'smstunnel'); ?>
                        <span class="count">(<?php echo isset($status_counts['sent']) ? intval($status_counts['sent']->count) : 0; ?>)</span>
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-history&status=failed')); ?>" <?php echo $status_filter === 'failed' ? 'class="current"' : ''; ?>>
                        <?php esc_html_e('Failed', 'smstunnel'); ?>
                        <span class="count">(<?php echo isset($status_counts['failed']) ? intval($status_counts['failed']->count) : 0; ?>)</span>
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-history&status=sending')); ?>" <?php echo $status_filter === 'sending' ? 'class="current"' : ''; ?>>
                        <?php esc_html_e('Pending', 'smstunnel'); ?>
                        <span class="count">(<?php echo isset($status_counts['sending']) ? intval($status_counts['sending']->count) : 0; ?>)</span>
                    </a>
                </li>
            </ul>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 50px;"><?php esc_html_e('ID', 'smstunnel'); ?></th>
                        <th style="width: 130px;"><?php esc_html_e('Phone', 'smstunnel'); ?></th>
                        <th><?php esc_html_e('Message', 'smstunnel'); ?></th>
                        <th style="width: 80px;"><?php esc_html_e('Status', 'smstunnel'); ?></th>
                        <th style="width: 80px;"><?php esc_html_e('Type', 'smstunnel'); ?></th>
                        <th style="width: 100px;"><?php esc_html_e('Source', 'smstunnel'); ?></th>
                        <th style="width: 140px;"><?php esc_html_e('Date', 'smstunnel'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($sms_log)): ?>
                    <tr>
                        <td colspan="7"><?php esc_html_e('No SMS messages found.', 'smstunnel'); ?></td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($sms_log as $sms): ?>
                    <tr>
                        <td><?php echo esc_html($sms->id); ?></td>
                        <td><?php echo esc_html($sms->phone); ?></td>
                        <td>
                            <?php echo esc_html(wp_trim_words($sms->message, 15)); ?>
                            <?php if (!empty($sms->error_message)): ?>
                                <br><small style="color: red;"><?php echo esc_html($sms->error_message); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php
                            $status_class = '';
                            switch ($sms->status) {
                                case 'sent': $status_class = 'color: green;'; break;
                                case 'failed': $status_class = 'color: red;'; break;
                                case 'sending': $status_class = 'color: orange;'; break;
                            }
                            ?>
                            <span style="<?php echo esc_attr($status_class); ?>">
                                <?php echo esc_html(ucfirst($sms->status)); ?>
                            </span>
                        </td>
                        <td><?php echo $sms->is_2fa ? '2FA' : 'SMS'; ?></td>
                        <td><?php echo esc_html($sms->source); ?></td>
                        <td><?php echo esc_html(date_i18n('Y-m-d H:i:s', strtotime($sms->created_at))); ?></td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <?php if ($total_pages > 1): ?>
            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <?php
                    echo wp_kses_post(paginate_links(array(
                        'base' => add_query_arg('paged', '%#%'),
                        'format' => '',
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                        'total' => $total_pages,
                        'current' => $current_page,
                    )));
                    ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render Inbox page (received SMS messages)
     */
    public function render_inbox_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $api = new SMSTunnel_API();

        // Pagination
        $per_page = 25;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Viewing data only
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Viewing data only
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

        // Filters
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Viewing data only
        $phone_filter = isset($_GET['phone']) ? sanitize_text_field(wp_unslash($_GET['phone'])) : '';

        // Fetch messages from API
        $result = $api->get_received_messages($per_page, $current_page, null, $phone_filter);

        $messages = array();
        $total_items = 0;
        $has_more = false;
        $error = '';

        if ($result['success']) {
            $messages = $result['messages'];
            $total_items = $result['total'];
            $has_more = $result['hasMore'];
        } else {
            $error = $result['error'];
        }

        $total_pages = ceil($total_items / $per_page);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Inbox - Received SMS', 'smstunnel'); ?></h1>

            <?php if (!empty($error)): ?>
            <div class="notice notice-error">
                <p><?php echo esc_html($error); ?></p>
            </div>
            <?php endif; ?>

            <div class="tablenav top">
                <form method="get" style="display: inline-block;">
                    <input type="hidden" name="page" value="smstunnel-inbox">
                    <label for="phone-filter"><?php esc_html_e('Filter by phone:', 'smstunnel'); ?></label>
                    <input type="text" id="phone-filter" name="phone" value="<?php echo esc_attr($phone_filter); ?>" placeholder="+40712345678" style="width: 150px;">
                    <input type="submit" class="button" value="<?php esc_html_e('Filter', 'smstunnel'); ?>">
                    <?php if (!empty($phone_filter)): ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-inbox')); ?>" class="button"><?php esc_html_e('Clear', 'smstunnel'); ?></a>
                    <?php endif; ?>
                </form>
                <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-inbox')); ?>" class="button" style="margin-left: 10px;">
                    <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                    <?php esc_html_e('Refresh', 'smstunnel'); ?>
                </a>
                <span style="margin-left: 15px; color: #666;">
                    <?php
                    /* translators: %d: number of messages */
                    printf(esc_html__('Total: %d messages', 'smstunnel'), intval($total_items));
                    ?>
                </span>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 140px;"><?php esc_html_e('From', 'smstunnel'); ?></th>
                        <th><?php esc_html_e('Message', 'smstunnel'); ?></th>
                        <th style="width: 130px;"><?php esc_html_e('Device', 'smstunnel'); ?></th>
                        <th style="width: 150px;"><?php esc_html_e('Received', 'smstunnel'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($messages)): ?>
                    <tr>
                        <td colspan="4"><?php esc_html_e('No received messages found.', 'smstunnel'); ?></td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($messages as $msg): ?>
                    <tr>
                        <td>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-inbox&phone=' . urlencode($msg['sender']))); ?>">
                                <?php echo esc_html($msg['sender']); ?>
                            </a>
                        </td>
                        <td>
                            <?php if ($msg['isEncrypted']): ?>
                                <em style="color: #999;">
                                    <span class="dashicons dashicons-lock" style="font-size: 14px; vertical-align: middle;"></span>
                                    <?php esc_html_e('[E2E Encrypted - Viewable only in mobile app]', 'smstunnel'); ?>
                                </em>
                            <?php else: ?>
                                <?php echo esc_html($msg['message']); ?>
                            <?php endif; ?>
                        </td>
                        <td><?php echo esc_html($msg['deviceName']); ?></td>
                        <td>
                            <?php
                            $received_at = strtotime($msg['receivedAt']);
                            echo esc_html(date_i18n('Y-m-d H:i', $received_at));
                            ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <?php if ($total_pages > 1): ?>
            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <?php
                    $base_url = admin_url('admin.php?page=smstunnel-inbox');
                    if (!empty($phone_filter)) {
                        $base_url .= '&phone=' . urlencode($phone_filter);
                    }
                    echo wp_kses_post(paginate_links(array(
                        'base' => add_query_arg('paged', '%#%', $base_url),
                        'format' => '',
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                        'total' => $total_pages,
                        'current' => $current_page,
                    )));
                    ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render Developer Docs page
     */
    public function render_docs_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Developer Documentation', 'smstunnel'); ?></h1>

            <div class="card" style="max-width: 800px; padding: 20px;">
                <h2><?php esc_html_e('Sending SMS from Your Code', 'smstunnel'); ?></h2>
                <p><?php esc_html_e('Use the global function to send SMS from anywhere in your WordPress code:', 'smstunnel'); ?></p>
                <pre style="background: #f1f1f1; padding: 15px; overflow-x: auto;">
// Basic usage
$result = smstunnel_send_sms('+40712345678', 'Hello from WordPress!');

// With options
$result = smstunnel_send_sms('+40712345678', 'Your code is 123456', array(
    'is_2fa'  => true,      // Mark as 2FA message (shorter timeout)
    'source'  => 'my-plugin' // Track source in SMS history
));

// Check result
if ($result['success']) {
    echo 'SMS sent! Message ID: ' . $result['messageId'];
} else {
    echo 'Error: ' . $result['error'];
}</pre>

                <h3 style="margin-top: 20px;"><?php esc_html_e('Using Action Hook', 'smstunnel'); ?></h3>
                <pre style="background: #f1f1f1; padding: 15px; overflow-x: auto;">
// Fire and forget - no return value
do_action('smstunnel_send_sms', '+40712345678', 'Hello!');</pre>
            </div>
        </div>
        <?php
    }

    /**
     * Admin scripts
     */
    public function admin_scripts($hook) {
        if (strpos($hook, 'smstunnel') === false) {
            return;
        }

        wp_enqueue_script('jquery');

        // Add inline script for test connection
        wp_add_inline_script('jquery', '
            jQuery(function($) {
                $("#smstunnel-test-connection").on("click", function() {
                    var $btn = $(this);
                    var $result = $("#smstunnel-test-result");

                    $btn.prop("disabled", true);
                    $result.html("<span style=\"color:blue;\">' . esc_js(__('Testing...', 'smstunnel')) . '</span>");

                    $.post(ajaxurl, {
                        action: "smstunnel_test_connection",
                        nonce: "' . wp_create_nonce('smstunnel_nonce') . '"
                    }, function(response) {
                        $btn.prop("disabled", false);
                        if (response.success) {
                            $result.html("<span style=\"color:green;\">" + response.data.message + "</span>");
                        } else {
                            $result.html("<span style=\"color:red;\">" + (response.data ? response.data.message : "Error") + "</span>");
                        }
                    }).fail(function() {
                        $btn.prop("disabled", false);
                        $result.html("<span style=\"color:red;\">' . esc_js(__('Request failed.', 'smstunnel')) . '</span>");
                    });
                });

                // API Key connect button
                var $apiKeyInput = $("#smstunnel-api-key-input");
                var $apiKeyConnect = $("#smstunnel-api-key-connect");
                var $apiKeyStatus = $("#smstunnel-api-key-status");
                var $apiKeyHidden = $("#smstunnel-api-key-hidden");

                // Enable/disable connect button based on input
                $apiKeyInput.on("input", function() {
                    var val = $(this).val().trim();
                    $apiKeyConnect.prop("disabled", val.length < 10);
                    $apiKeyStatus.html("");
                });

                // Connect button click
                $apiKeyConnect.on("click", function() {
                    var apiKey = $apiKeyInput.val().trim();
                    if (!apiKey) return;

                    $apiKeyConnect.prop("disabled", true);
                    $apiKeyStatus.html("<span style=\"color:blue;\">' . esc_js(__('Verifying...', 'smstunnel')) . '</span>");

                    $.post(ajaxurl, {
                        action: "smstunnel_verify_api_key",
                        nonce: "' . wp_create_nonce('smstunnel_nonce') . '",
                        api_key: apiKey
                    }, function(response) {
                        if (response.success) {
                            $apiKeyStatus.html("<span style=\"color:green;\">' . esc_js(__('Connected successfully!', 'smstunnel')) . '</span>");
                            // Reload page to show connected state
                            setTimeout(function() {
                                window.location.reload();
                            }, 1000);
                        } else {
                            $apiKeyConnect.prop("disabled", false);
                            $apiKeyStatus.html("<span style=\"color:red;\">" + (response.data ? response.data.message : "' . esc_js(__('Verification failed', 'smstunnel')) . '") + "</span>");
                        }
                    }).fail(function() {
                        $apiKeyConnect.prop("disabled", false);
                        $apiKeyStatus.html("<span style=\"color:red;\">' . esc_js(__('Request failed.', 'smstunnel')) . '</span>");
                    });
                });
            });
        ');

        // Social Login Script (uses backend redirect, no SDKs needed)
        wp_enqueue_script(
            'smstunnel-social',
            SMSTUNNEL_PLUGIN_URL . 'assets/js/social-login.js',
            array('jquery'),
            SMSTUNNEL_VERSION . '.' . time(), // Cache bust during development
            true
        );

        // Callback URL for OAuth redirect
        $callback_url = admin_url('admin.php?page=smstunnel&smstunnel_oauth_callback=1');

        wp_localize_script('smstunnel-social', 'smstunnelSocial', array(
            'callbackUrl' => $callback_url,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('smstunnel_social_login'),
        ));

        // Define smstunnel_admin early (on jQuery which is always loaded in admin)
        wp_add_inline_script('jquery', '
            window.smstunnel_admin = {
                ajaxUrl: "' . esc_js(admin_url('admin-ajax.php')) . '",
                nonce: "' . esc_js(wp_create_nonce('smstunnel_nonce')) . '"
            };
        ');

        // E2E Pairing inline script
        wp_add_inline_script('jquery', '
            jQuery(function($) {
                var pairingToken = null;
                var pollInterval = null;

                // Start E2E Pairing button
                $("#smstunnel-start-e2e").on("click", function() {
                    var $btn = $(this);
                    var $qrContainer = $("#smstunnel-e2e-qr");
                    var $status = $("#smstunnel-e2e-status");

                    $btn.prop("disabled", true).text("' . esc_js(__('Starting...', 'smstunnel')) . '");
                    $status.html("");

                    $.ajax({
                        url: smstunnel_admin.ajaxUrl,
                        type: "POST",
                        data: {
                            action: "smstunnel_e2e_start",
                            nonce: smstunnel_admin.nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                pairingToken = response.data.token;
                                var qrData = response.data.qrData;

                                // Show QR container with image from QR API
                                var qrImageUrl = "https://api.qrserver.com/v1/create-qr-code/?size=250x250&data=" + encodeURIComponent(qrData);
                                $qrContainer.html(\'<div style="background: white; padding: 15px; display: inline-block; border-radius: 8px;"><img src="\' + qrImageUrl + \'" alt="QR Code" style="display: block;"/></div>\');
                                $qrContainer.show();

                                $status.html(
                                    \'<div style="padding: 10px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; margin-top: 10px;">\' +
                                    \'<strong>' . esc_js(__('Waiting for phone...', 'smstunnel')) . '</strong><br>\' +
                                    \'<span style="font-size: 12px;">' . esc_js(__('Open SMSTunnel app → Settings → Encryption Settings → Scan E2E QR', 'smstunnel')) . '</span>\' +
                                    \'</div>\'
                                );

                                $btn.text("' . esc_js(__('Cancel Pairing', 'smstunnel')) . '").prop("disabled", false);
                                $btn.off("click").on("click", cancelPairing);

                                // Start polling for completion
                                startPolling();
                            } else {
                                $status.html(\'<span style="color: red;">\' + response.data.message + \'</span>\');
                                $btn.text("' . esc_js(__('Start E2E Pairing', 'smstunnel')) . '").prop("disabled", false);
                            }
                        },
                        error: function() {
                            $status.html(\'<span style="color: red;">' . esc_js(__('Request failed. Please try again.', 'smstunnel')) . '</span>\');
                            $btn.text("' . esc_js(__('Start E2E Pairing', 'smstunnel')) . '").prop("disabled", false);
                        }
                    });
                });

                function startPolling() {
                    pollInterval = setInterval(function() {
                        $.ajax({
                            url: smstunnel_admin.ajaxUrl,
                            type: "POST",
                            data: {
                                action: "smstunnel_e2e_status",
                                nonce: smstunnel_admin.nonce,
                                token: pairingToken
                            },
                            success: function(response) {
                                if (response.success) {
                                    if (response.data.status === "completed") {
                                        // Success! Pairing complete
                                        stopPolling();
                                        $("#smstunnel-e2e-qr").hide();
                                        $("#smstunnel-e2e-status").html(
                                            \'<div style="padding: 15px; background: #e7f5e7; border: 1px solid #46b450; border-radius: 8px;">\' +
                                            \'<strong style="color: #2e7d32;">✓ ' . esc_js(__('E2E Pairing Complete!', 'smstunnel')) . '</strong><br>\' +
                                            \'<span style="font-size: 12px;">' . esc_js(__('Device:', 'smstunnel')) . ' \' + (response.data.deviceName || "Phone") + \'</span><br>\' +
                                            \'<code style="font-size: 11px;">' . esc_js(__('Fingerprint:', 'smstunnel')) . ' \' + response.data.fingerprint + \'</code>\' +
                                            \'</div>\' +
                                            \'<p style="margin-top: 10px; color: #2e7d32; font-weight: bold;">' . esc_js(__('Refreshing page...', 'smstunnel')) . '</p>\'
                                        );
                                        // Reload page to show active state
                                        setTimeout(function() {
                                            window.location.reload();
                                        }, 1500);
                                    }
                                } else {
                                    if (response.data.expired) {
                                        stopPolling();
                                        cancelPairing();
                                        $("#smstunnel-e2e-status").html(\'<span style="color: red;">' . esc_js(__('Pairing expired. Please try again.', 'smstunnel')) . '</span>\');
                                    }
                                }
                            }
                        });
                    }, 2000); // Poll every 2 seconds
                }

                function stopPolling() {
                    if (pollInterval) {
                        clearInterval(pollInterval);
                        pollInterval = null;
                    }
                }

                function cancelPairing() {
                    stopPolling();
                    pairingToken = null;
                    $("#smstunnel-e2e-qr").hide().html("");
                    $("#smstunnel-e2e-status").html("");
                    var $btn = $("#smstunnel-start-e2e");
                    $btn.html(\'<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm-6 9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2zm3.1-9H8.9V6c0-1.71 1.39-3.1 3.1-3.1 1.71 0 3.1 1.39 3.1 3.1v2z"/></svg> ' . esc_js(__('Start E2E Pairing', 'smstunnel')) . '\');
                    $btn.off("click").on("click", function() { window.location.reload(); });
                }

                // Remove E2E pairing
                $("#smstunnel-remove-e2e").on("click", function() {
                    if (!confirm("' . esc_js(__('Are you sure you want to remove E2E encryption? Messages will no longer be encrypted.', 'smstunnel')) . '")) {
                        return;
                    }

                    var $btn = $(this);
                    $btn.prop("disabled", true).text("' . esc_js(__('Removing...', 'smstunnel')) . '");

                    $.ajax({
                        url: smstunnel_admin.ajaxUrl,
                        type: "POST",
                        data: {
                            action: "smstunnel_e2e_remove",
                            nonce: smstunnel_admin.nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                window.location.reload();
                            } else {
                                alert(response.data.message || "Error");
                                $btn.prop("disabled", false).text("' . esc_js(__('Remove E2E Pairing', 'smstunnel')) . '");
                            }
                        },
                        error: function() {
                            alert("Request failed");
                            $btn.prop("disabled", false).text("' . esc_js(__('Remove E2E Pairing', 'smstunnel')) . '");
                        }
                    });
                });
            });
        ');
    }

    /**
     * AJAX: Send SMS
     */
    public function ajax_send_sms() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';

        if (empty($phone) || empty($message)) {
            wp_send_json_error(array('message' => __('Phone and message are required.', 'smstunnel')));
        }

        $result = smstunnel_send_sms($phone, $message, array('source' => 'admin-panel'));

        if ($result['success']) {
            wp_send_json_success(array('message' => __('SMS sent successfully!', 'smstunnel')));
        } else {
            wp_send_json_error(array('message' => $result['error']));
        }
    }

    /**
     * AJAX: Test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $api = new SMSTunnel_API();
        $result = $api->test_connection();

        if ($result['success']) {
            $message = sprintf(
                /* translators: %1$d: total devices, %2$d: online devices */
                __('Connected! %1$d device(s) found, %2$d online.', 'smstunnel'),
                $result['total'],
                $result['online']
            );
            wp_send_json_success(array('message' => $message));
        } else {
            wp_send_json_error(array('message' => $result['error']));
        }
    }

    /**
     * AJAX: Verify API key
     */
    public function ajax_verify_api_key() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';

        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key is required.', 'smstunnel')));
        }

        // Verify the API key by calling the server
        $response = wp_remote_get(
            'https://smstunnel.io/api/devices',
            array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json',
                ),
                'timeout' => 15,
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection failed: ', 'smstunnel') . $response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 200) {
            // API key is valid - save it and fetch user info
            update_option('smstunnel_api_key', $api_key);

            // Try to get user info
            $user_response = wp_remote_get(
                'https://smstunnel.io/auth/me',
                array(
                    'headers' => array(
                        'Authorization' => 'Bearer ' . $api_key,
                    ),
                    'timeout' => 10,
                )
            );

            if (!is_wp_error($user_response) && wp_remote_retrieve_response_code($user_response) === 200) {
                $user_data = json_decode(wp_remote_retrieve_body($user_response), true);
                if (!empty($user_data)) {
                    update_option('smstunnel_user', array(
                        'email' => $user_data['email'] ?? '',
                        'firstName' => $user_data['firstName'] ?? '',
                        'lastName' => $user_data['lastName'] ?? '',
                    ));
                }
            }

            wp_send_json_success(array('message' => __('API key verified successfully!', 'smstunnel')));
        } elseif ($status_code === 401 || $status_code === 403) {
            wp_send_json_error(array('message' => __('Invalid API key.', 'smstunnel')));
        } else {
            wp_send_json_error(array('message' => __('Verification failed. Status: ', 'smstunnel') . $status_code));
        }
    }

    /**
     * AJAX: Start E2E pairing - creates pairing request and returns QR data
     */
    public function ajax_e2e_start() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $api_key = get_option('smstunnel_api_key', '');
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key not configured.', 'smstunnel')));
        }

        $server_url = get_option('smstunnel_server_url', '');
        if (empty($server_url)) {
            $server_url = defined('SMSTUNNEL_API_BASE') ? SMSTUNNEL_API_BASE : 'https://smstunnel.io/api/v1';
        }
        $server_url = rtrim($server_url, '/');

        // Call server to create E2E pairing request
        $response = wp_remote_post(
            $server_url . '/e2e/pair',
            array(
                'headers' => array(
                    'X-API-Key' => $api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'siteName' => get_bloginfo('name'),
                    'siteUrl' => get_site_url(),
                )),
                'timeout' => 15,
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection failed: ', 'smstunnel') . $response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 200 && !empty($body['success'])) {
            wp_send_json_success(array(
                'token' => $body['data']['token'],
                'qrData' => $body['data']['qrData'],
                'expiresAt' => $body['data']['expiresAt'],
            ));
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : __('Failed to start pairing', 'smstunnel');
            wp_send_json_error(array('message' => $error_msg));
        }
    }

    /**
     * AJAX: Check E2E pairing status
     */
    public function ajax_e2e_status() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $token = isset($_POST['token']) ? sanitize_text_field(wp_unslash($_POST['token'])) : '';
        if (empty($token)) {
            wp_send_json_error(array('message' => __('Token is required.', 'smstunnel')));
        }

        $api_key = get_option('smstunnel_api_key', '');
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key not configured.', 'smstunnel')));
        }

        $server_url = get_option('smstunnel_server_url', '');
        if (empty($server_url)) {
            $server_url = defined('SMSTUNNEL_API_BASE') ? SMSTUNNEL_API_BASE : 'https://smstunnel.io/api/v1';
        }
        $server_url = rtrim($server_url, '/');

        // Check pairing status
        $response = wp_remote_get(
            $server_url . '/e2e/pair/' . $token,
            array(
                'headers' => array(
                    'X-API-Key' => $api_key,
                ),
                'timeout' => 10,
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection failed: ', 'smstunnel') . $response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 200 && !empty($body['success'])) {
            $data = $body['data'];

            if ($data['status'] === 'completed') {
                // Save single E2E key
                $e2e_key = array(
                    'device_id' => $data['deviceId'],
                    'public_key' => $data['publicKey'],
                    'device_name' => $data['deviceName'] ?: __('Phone', 'smstunnel'),
                    'paired_at' => current_time('mysql'),
                );

                update_option('smstunnel_e2e_key', $e2e_key);

                wp_send_json_success(array(
                    'status' => 'completed',
                    'deviceName' => $data['deviceName'],
                    'fingerprint' => substr(hash('sha256', $data['publicKey']), 0, 16),
                ));
            } else {
                wp_send_json_success(array(
                    'status' => $data['status'],
                ));
            }
        } elseif ($status_code === 404) {
            wp_send_json_error(array('message' => __('Pairing expired or not found.', 'smstunnel'), 'expired' => true));
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : __('Failed to check status', 'smstunnel');
            wp_send_json_error(array('message' => $error_msg));
        }
    }

    /**
     * AJAX: Remove E2E pairing
     */
    public function ajax_e2e_remove() {
        check_ajax_referer('smstunnel_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        delete_option('smstunnel_e2e_key');

        wp_send_json_success();
    }

    /**
     * Handle OAuth callback from smstunnel.io
     */
    public function handle_oauth_callback() {
        // Check if this is an OAuth callback
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external service
        if (!isset($_GET['page']) || $_GET['page'] !== 'smstunnel') {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external service
        if (!isset($_GET['smstunnel_oauth_callback'])) {
            return;
        }

        // Get tokens from URL parameters
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external service
        $access_token = isset($_GET['accessToken']) ? sanitize_text_field(wp_unslash($_GET['accessToken'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external service
        $refresh_token = isset($_GET['refreshToken']) ? sanitize_text_field(wp_unslash($_GET['refreshToken'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external service
        $error = isset($_GET['error']) ? sanitize_text_field(wp_unslash($_GET['error'])) : '';

        if (!empty($error)) {
            // Redirect with error
            wp_safe_redirect(admin_url('admin.php?page=smstunnel&smstunnel_auth_error=' . urlencode($error)));
            exit;
        }

        if (!empty($access_token)) {
            // Save tokens
            update_option('smstunnel_access_token', $access_token);

            if (!empty($refresh_token)) {
                update_option('smstunnel_refresh_token', $refresh_token);
            }

            // Fetch user profile from backend
            $response = wp_remote_get('https://smstunnel.io/auth/me', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
                'timeout' => 15,
                'sslverify' => true,
            ));

            $user_saved = false;
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $body = json_decode(wp_remote_retrieve_body($response), true);
                if (!empty($body) && (isset($body['email']) || isset($body['userId']))) {
                    update_option('smstunnel_user', $body);
                    $user_saved = true;
                }
            }

            // Create a new API key for WordPress using the Public API
            $api_key = $this->fetch_or_create_api_key($access_token);

            // Save API key
            if (!empty($api_key)) {
                update_option('smstunnel_api_key', $api_key);
            }

            // Redirect with success (even if API key wasn't fetched, user is authenticated)
            wp_safe_redirect(admin_url('admin.php?page=smstunnel&smstunnel_auth=success'));
            exit;
        }

        // No tokens received
        wp_safe_redirect(admin_url('admin.php?page=smstunnel&smstunnel_auth_error=' . urlencode('No authentication data received')));
        exit;
    }

    /**
     * Handle disconnect action
     */
    public function handle_disconnect() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'smstunnel') {
            return;
        }

        if (!isset($_GET['smstunnel_disconnect']) || $_GET['smstunnel_disconnect'] !== '1') {
            return;
        }

        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'smstunnel_disconnect')) {
            wp_die(esc_html__('Security check failed.', 'smstunnel'));
        }

        // Clear all authentication data
        delete_option('smstunnel_api_key');
        delete_option('smstunnel_access_token');
        delete_option('smstunnel_refresh_token');
        delete_option('smstunnel_user');

        // Redirect back to settings
        wp_safe_redirect(admin_url('admin.php?page=smstunnel&smstunnel_disconnected=1'));
        exit;
    }

    /**
     * AJAX: Social Login (Google/Facebook)
     * Authenticates with smstunnel.io using social provider tokens
     */
    public function ajax_social_login() {
        check_ajax_referer('smstunnel_social_login', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $provider = isset($_POST['provider']) ? sanitize_text_field(wp_unslash($_POST['provider'])) : '';
        $id_token = isset($_POST['idToken']) ? sanitize_text_field(wp_unslash($_POST['idToken'])) : '';
        $access_token = isset($_POST['accessToken']) ? sanitize_text_field(wp_unslash($_POST['accessToken'])) : '';

        if (empty($provider) || ($provider === 'google' && empty($id_token)) || ($provider === 'facebook' && empty($access_token))) {
            wp_send_json_error(array('message' => __('Invalid authentication data.', 'smstunnel')));
        }

        // Call smstunnel.io /auth/social endpoint
        $response = wp_remote_post('https://smstunnel.io/auth/social', array(
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'provider' => $provider,
                'idToken' => $id_token,
                'accessToken' => $access_token,
            )),
            'timeout' => 30,
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 200 || $status_code === 201) {
            // Save tokens and user info
            if (!empty($body['accessToken'])) {
                update_option('smstunnel_access_token', sanitize_text_field($body['accessToken']));
            }
            if (!empty($body['refreshToken'])) {
                update_option('smstunnel_refresh_token', sanitize_text_field($body['refreshToken']));
            }
            if (!empty($body['user'])) {
                update_option('smstunnel_user', $body['user']);
            }

            // Create API key using the access token
            $jwt_token = $body['accessToken'] ?? '';
            if (!empty($jwt_token)) {
                $api_key = $this->fetch_or_create_api_key($jwt_token);
                if (!empty($api_key)) {
                    update_option('smstunnel_api_key', $api_key);
                }
            }

            wp_send_json_success(array(
                'message' => __('Connected successfully!', 'smstunnel'),
                'user' => $body['user'] ?? null,
            ));
        } else {
            $error_message = isset($body['message']) ? $body['message'] : __('Authentication failed.', 'smstunnel');
            wp_send_json_error(array('message' => $error_message));
        }
    }

    /**
     * AJAX: Email/Password Login
     * Authenticates with smstunnel.io using email and password
     */
    public function ajax_email_login() {
        check_ajax_referer('smstunnel_social_login', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'smstunnel')));
        }

        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Password should not be modified
        $password = isset($_POST['password']) ? wp_unslash($_POST['password']) : '';

        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => __('Please enter email and password.', 'smstunnel')));
        }

        // Call smstunnel.io /api/v1/auth/login endpoint
        $response = wp_remote_post('https://smstunnel.io/api/v1/auth/login', array(
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'email' => $email,
                'password' => $password,
            )),
            'timeout' => 30,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 200 || $status_code === 201) {
            // New API response format: { success: true, data: { accessToken, user, ... } }
            $data = isset($body['data']) ? $body['data'] : $body;

            // Save tokens
            if (!empty($data['accessToken'])) {
                update_option('smstunnel_access_token', sanitize_text_field($data['accessToken']));
            }
            if (!empty($data['refreshToken'])) {
                update_option('smstunnel_refresh_token', sanitize_text_field($data['refreshToken']));
            }
            if (!empty($data['user'])) {
                update_option('smstunnel_user', $data['user']);
            }

            // Fetch or create API key using the JWT token
            $access_token = $data['accessToken'] ?? '';
            if (!empty($access_token)) {
                $api_key = $this->fetch_or_create_api_key($access_token);
                if (!empty($api_key)) {
                    update_option('smstunnel_api_key', $api_key);
                }
            }

            wp_send_json_success(array(
                'message' => __('Connected successfully!', 'smstunnel'),
                'user' => $data['user'] ?? null,
            ));
        } else {
            $error_message = isset($body['message']) ? $body['message'] : __('Invalid email or password.', 'smstunnel');
            wp_send_json_error(array('message' => $error_message));
        }
    }

    /**
     * Helper: Create API key using JWT token
     * New endpoint: POST /api/v1/api-keys
     * The plainKey is returned only once and must be saved
     */
    private function fetch_or_create_api_key($access_token) {
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');

        // Create new API key for WordPress using Public API
        $create_response = wp_remote_post('https://smstunnel.io/api/v1/api-keys', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'name' => 'WordPress Plugin - ' . $site_name,
                'metadata' => array(
                    'siteUrl' => $site_url,
                    'siteName' => $site_name,
                    'source' => 'wordpress-plugin',
                ),
            )),
            'timeout' => 15,
            'sslverify' => true,
        ));

        if (!is_wp_error($create_response)) {
            $status = wp_remote_retrieve_response_code($create_response);
            if ($status === 200 || $status === 201) {
                $response_body = json_decode(wp_remote_retrieve_body($create_response), true);
                // New response format: { "success": true, "data": { "apiKey": {...}, "plainKey": "sk_live_..." } }
                if (!empty($response_body['data']['plainKey'])) {
                    return $response_body['data']['plainKey'];
                }
                // Fallback for old format
                if (!empty($response_body['apiKey']['key'])) {
                    return $response_body['apiKey']['key'];
                }
            }
        }

        return '';
    }

    /**
     * WordPress Login 2FA - Add form fields
     */
    public function wp_2fa_login_form() {
        ?>
        <p id="smstunnel-2fa-field" style="display:none;">
            <label for="smstunnel_2fa_code" style="display:block; margin-bottom:8px;"><?php esc_html_e('Verification Code', 'smstunnel'); ?></label>
            <span id="smstunnel-2fa-input-wrapper" style="display:flex; align-items:stretch;">
                <input type="text" name="smstunnel_2fa_code" id="smstunnel_2fa_code" class="input" style="width:100%;" autocomplete="off" maxlength="<?php echo intval(get_option('smstunnel_2fa_length', 6)); ?>" />
            </span>
        </p>
        <?php if (get_option('smstunnel_2fa_remember_device', 'no') === 'yes') : ?>
        <p id="smstunnel-remember-device-field" style="display:none; margin-top:10px;">
            <label>
                <input type="checkbox" name="smstunnel_remember_device" id="smstunnel_remember_device" value="1" />
                <?php
                /* translators: %d: number of days */
                printf(esc_html__('Remember this device for %d days', 'smstunnel'), intval(get_option('smstunnel_2fa_remember_days', 30)));
                ?>
            </label>
        </p>
        <?php endif; ?>
        <p id="smstunnel-2fa-status" style="display:none; color: green; margin-bottom: 16px;"></p>
        <!-- Phone Setup Form (hidden by default) -->
        <div id="smstunnel-phone-setup" style="display: none; margin-bottom: 16px;">
            <p id="smstunnel-phone-setup-message" style="margin-bottom: 10px;"></p>

            <div id="smstunnel-phone-input-group">
                <label for="smstunnel-phone-input"><?php esc_html_e('Phone Number', 'smstunnel'); ?></label>
                <input type="tel" id="smstunnel-phone-input"
                       placeholder="+40712345678"
                       style="width: 100%; margin: 5px 0 10px 0; padding: 8px;">
                <button type="button" id="smstunnel-send-phone-code" class="button button-primary" style="width: 100%;">
                    <?php esc_html_e('Send Verification Code', 'smstunnel'); ?>
                </button>
            </div>

            <div id="smstunnel-phone-code-group" style="display: none; margin-top: 15px;">
                <label for="smstunnel-phone-code"><?php esc_html_e('Verification Code', 'smstunnel'); ?></label>
                <input type="text" id="smstunnel-phone-code"
                       placeholder="123456"
                       style="width: 100%; margin: 5px 0 10px 0; padding: 8px; text-align: center; font-size: 18px; letter-spacing: 5px;">
                <button type="button" id="smstunnel-verify-phone-code" class="button button-primary" style="width: 100%;">
                    <?php esc_html_e('Verify Code', 'smstunnel'); ?>
                </button>
            </div>

            <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #ddd;">
                <button type="button" id="smstunnel-skip-phone-setup" class="button" style="width: 100%;">
                    <?php esc_html_e('Skip - I don\'t have my phone', 'smstunnel'); ?>
                </button>
                <p style="font-size: 12px; color: #666; margin-top: 5px;">
                    <?php esc_html_e('You will need to add your phone number later from your profile.', 'smstunnel'); ?>
                </p>
            </div>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var form = document.getElementById('loginform');
            var userField = document.getElementById('user_login');
            var passField = document.getElementById('user_pass');
            var codeField = document.getElementById('smstunnel_2fa_code');
            var statusEl = document.getElementById('smstunnel-2fa-status');
            var twoFAField = document.getElementById('smstunnel-2fa-field');
            var rememberDeviceField = document.getElementById('smstunnel-remember-device-field');
            var submitBtn = document.getElementById('wp-submit');
            var codeSent = false;
            var requiresCode = false;
            var currentXhr = null;
            var originalBtnText = submitBtn ? submitBtn.value : '';
            var setupToken = '';
            var phoneSetupEl = document.getElementById('smstunnel-phone-setup');
            var phoneSetupMsg = document.getElementById('smstunnel-phone-setup-message');
            var phoneInputGroup = document.getElementById('smstunnel-phone-input-group');
            var phoneCodeGroup = document.getElementById('smstunnel-phone-code-group');
            var phoneInput = document.getElementById('smstunnel-phone-input');
            var phoneCodeInput = document.getElementById('smstunnel-phone-code');
            var sendPhoneCodeBtn = document.getElementById('smstunnel-send-phone-code');
            var verifyPhoneCodeBtn = document.getElementById('smstunnel-verify-phone-code');
            var skipPhoneSetupBtn = document.getElementById('smstunnel-skip-phone-setup');

            if (!form || !submitBtn) {
                console.error('SMSTunnel 2FA: Form or submit button not found');
                return;
            }

            function setLoading(isLoading) {
                if (isLoading) {
                    submitBtn.disabled = true;
                    submitBtn.value = '<?php esc_html_e('Checking...', 'smstunnel'); ?> ⏳';
                    submitBtn.style.opacity = '0.7';
                } else {
                    submitBtn.disabled = false;
                    submitBtn.value = originalBtnText;
                    submitBtn.style.opacity = '1';
                }
            }

            form.addEventListener('submit', function(e) {
                if (requiresCode && !codeField.value.trim()) {
                    e.preventDefault();
                    statusEl.style.color = 'red';
                    statusEl.textContent = '<?php esc_html_e('Please enter the verification code', 'smstunnel'); ?>';
                    statusEl.style.display = 'block';
                    return false;
                }

                if (!codeSent && userField.value.trim() && passField.value.trim()) {
                    e.preventDefault();

                    setLoading(true);
                    statusEl.style.color = 'blue';
                    statusEl.textContent = '<?php esc_html_e('Verifying credentials...', 'smstunnel'); ?>';
                    statusEl.style.display = 'block';

                    currentXhr = new XMLHttpRequest();
                    currentXhr.open('POST', '<?php echo esc_url(admin_url('admin-ajax.php')); ?>', true);
                    currentXhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    currentXhr.timeout = 25000;

                    currentXhr.ontimeout = function() {
                        setLoading(false);
                        codeSent = false;
                        requiresCode = false;
                        statusEl.style.color = 'red';
                        statusEl.textContent = '<?php esc_html_e('Request timeout. Please try again.', 'smstunnel'); ?>';
                    };

                    currentXhr.onerror = function() {
                        setLoading(false);
                        codeSent = false;
                        requiresCode = false;
                        statusEl.style.color = 'red';
                        statusEl.textContent = '<?php esc_html_e('Connection error. Please try again.', 'smstunnel'); ?>';
                    };

                    currentXhr.onload = function() {
                        if (currentXhr.status === 200) {
                            var res = JSON.parse(currentXhr.responseText);
                            if (res.success) {
                                if (res.data.requires_2fa) {
                                    codeSent = true;
                                    requiresCode = true;
                                    setLoading(false);
                                    if (res.data.prefix) {
                                        var wrapper = document.getElementById('smstunnel-2fa-input-wrapper');
                                        var prefixSpan = document.createElement('span');
                                        prefixSpan.id = 'smstunnel-prefix-display';
                                        prefixSpan.style.cssText = 'display:inline-flex; align-items:center; font-size:16px; font-weight:bold; padding:0 10px; background:#f0f0f1; border:1px solid #8c8f94; border-right:none; border-radius:4px 0 0 4px; white-space:nowrap;';
                                        prefixSpan.textContent = res.data.prefix + ' -';
                                        wrapper.insertBefore(prefixSpan, codeField);
                                        codeField.style.cssText = 'flex:1; min-width:100px; border-radius:0 4px 4px 0; margin:0;';
                                    }
                                    twoFAField.style.display = 'block';
                                    if (rememberDeviceField) {
                                        rememberDeviceField.style.display = 'block';
                                    }
                                    codeField.focus();
                                    statusEl.style.color = 'green';
                                    statusEl.innerHTML = res.data.message;
                                    statusEl.style.display = 'block';
                                } else if (res.data.bypass) {
                                    submitBtn.value = '<?php esc_html_e('Logging in...', 'smstunnel'); ?>';
                                    statusEl.style.color = 'orange';
                                    statusEl.innerHTML = res.data.message || '<?php esc_html_e('SMS server unavailable - logging in without 2FA', 'smstunnel'); ?>';
                                    statusEl.style.display = 'block';
                                    codeSent = true;
                                    setTimeout(function() { form.submit(); }, 500);
                                } else if (res.data.requires_phone_setup) {
                                    // Phone setup required
                                    setupToken = res.data.setup_token;
                                    setLoading(false);
                                    phoneSetupEl.style.display = 'block';
                                    phoneSetupMsg.innerHTML = res.data.message;
                                    submitBtn.style.display = 'none';
                                    statusEl.style.display = 'none';
                                    phoneInput.focus();
                                } else {
                                    codeSent = true;
                                    form.submit();
                                }
                            } else {
                                setLoading(false);
                                codeSent = false;
                                requiresCode = false;
                                twoFAField.style.display = 'none';
                                statusEl.style.color = 'red';
                                statusEl.innerHTML = res.data ? res.data.message : 'Error';
                                statusEl.style.display = 'block';
                            }
                        } else {
                            setLoading(false);
                            codeSent = false;
                            requiresCode = false;
                            twoFAField.style.display = 'none';
                            statusEl.style.color = 'red';
                            statusEl.textContent = '<?php esc_html_e('Server error. Please try again.', 'smstunnel'); ?>';
                            statusEl.style.display = 'block';
                        }
                    };
                    currentXhr.send('action=smstunnel_wp_send_2fa&username=' + encodeURIComponent(userField.value) + '&password=' + encodeURIComponent(passField.value));
                }
            });

            // Handler for "Send Verification Code" button
            if (sendPhoneCodeBtn) {
                sendPhoneCodeBtn.addEventListener('click', function() {
                    var phone = phoneInput.value.trim();
                    if (!phone) {
                        phoneSetupMsg.innerHTML = '<span style="color: #dc3232;"><?php esc_html_e('Please enter your phone number.', 'smstunnel'); ?></span>';
                        return;
                    }

                    var btn = this;
                    btn.disabled = true;
                    btn.textContent = '<?php esc_html_e('Sending...', 'smstunnel'); ?>';

                    var xhr = new XMLHttpRequest();
                    xhr.open('POST', '<?php echo esc_url(admin_url('admin-ajax.php')); ?>');
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.onreadystatechange = function() {
                        if (xhr.readyState === 4) {
                            btn.disabled = false;
                            btn.textContent = '<?php esc_html_e('Send Verification Code', 'smstunnel'); ?>';
                            try {
                                var resp = JSON.parse(xhr.responseText);
                                if (resp.success) {
                                    phoneInputGroup.style.display = 'none';
                                    phoneCodeGroup.style.display = 'block';
                                    phoneSetupMsg.innerHTML = '<span style="color: #46b450;">' + resp.data.message + '</span>';
                                    phoneCodeInput.focus();
                                } else {
                                    phoneSetupMsg.innerHTML = '<span style="color: #dc3232;">' + resp.data.message + '</span>';
                                }
                            } catch (e) {
                                phoneSetupMsg.innerHTML = '<span style="color: #dc3232;"><?php esc_html_e('Server error. Please try again.', 'smstunnel'); ?></span>';
                            }
                        }
                    };
                    xhr.send('action=smstunnel_verify_phone_setup&setup_token=' + encodeURIComponent(setupToken) + '&phone=' + encodeURIComponent(phone));
                });
            }

            // Handler for "Verify Code" button
            if (verifyPhoneCodeBtn) {
                verifyPhoneCodeBtn.addEventListener('click', function() {
                    var code = phoneCodeInput.value.trim();
                    if (!code) {
                        phoneSetupMsg.innerHTML = '<span style="color: #dc3232;"><?php esc_html_e('Please enter the verification code.', 'smstunnel'); ?></span>';
                        return;
                    }

                    var btn = this;
                    btn.disabled = true;
                    btn.textContent = '<?php esc_html_e('Verifying...', 'smstunnel'); ?>';

                    var xhr = new XMLHttpRequest();
                    xhr.open('POST', '<?php echo esc_url(admin_url('admin-ajax.php')); ?>');
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.onreadystatechange = function() {
                        if (xhr.readyState === 4) {
                            btn.disabled = false;
                            btn.textContent = '<?php esc_html_e('Verify Code', 'smstunnel'); ?>';
                            try {
                                var resp = JSON.parse(xhr.responseText);
                                if (resp.success && resp.data.phone_saved) {
                                    phoneSetupMsg.innerHTML = '<span style="color: #46b450;">' + resp.data.message + '</span>';
                                    // Submit login form after short delay
                                    setTimeout(function() {
                                        codeSent = true;
                                        form.submit();
                                    }, 1000);
                                } else {
                                    phoneSetupMsg.innerHTML = '<span style="color: #dc3232;">' + resp.data.message + '</span>';
                                }
                            } catch (e) {
                                phoneSetupMsg.innerHTML = '<span style="color: #dc3232;"><?php esc_html_e('Server error. Please try again.', 'smstunnel'); ?></span>';
                            }
                        }
                    };
                    xhr.send('action=smstunnel_confirm_phone_setup&setup_token=' + encodeURIComponent(setupToken) + '&code=' + encodeURIComponent(code));
                });
            }

            // Handler for "Skip" button
            if (skipPhoneSetupBtn) {
                skipPhoneSetupBtn.addEventListener('click', function() {
                    var btn = this;
                    btn.disabled = true;
                    btn.textContent = '<?php esc_html_e('Skipping...', 'smstunnel'); ?>';

                    var xhr = new XMLHttpRequest();
                    xhr.open('POST', '<?php echo esc_url(admin_url('admin-ajax.php')); ?>');
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.onreadystatechange = function() {
                        if (xhr.readyState === 4) {
                            btn.disabled = false;
                            btn.textContent = '<?php echo esc_js(__('Skip - I don\'t have my phone', 'smstunnel')); ?>';
                            try {
                                var resp = JSON.parse(xhr.responseText);
                                if (resp.success && resp.data.skipped) {
                                    phoneSetupMsg.innerHTML = '<span style="color: #46b450;">' + resp.data.message + '</span>';
                                    // Submit login form after short delay
                                    setTimeout(function() {
                                        codeSent = true;
                                        form.submit();
                                    }, 1000);
                                } else {
                                    phoneSetupMsg.innerHTML = '<span style="color: #dc3232;">' + (resp.data ? resp.data.message : '<?php esc_html_e('Error. Please try again.', 'smstunnel'); ?>') + '</span>';
                                }
                            } catch (e) {
                                phoneSetupMsg.innerHTML = '<span style="color: #dc3232;"><?php esc_html_e('Server error. Please try again.', 'smstunnel'); ?></span>';
                            }
                        }
                    };
                    xhr.send('action=smstunnel_skip_phone_setup&setup_token=' + encodeURIComponent(setupToken));
                });
            }
        });
        </script>
        <?php
    }

    /**
     * WordPress Login 2FA - AJAX send code
     */
    public function ajax_wp_send_2fa() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called from login form AJAX
        $username = isset($_POST['username']) ? sanitize_user(wp_unslash($_POST['username'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Password
        $password = isset($_POST['password']) ? wp_unslash($_POST['password']) : '';

        if (empty($username) || empty($password)) {
            wp_send_json_error(array('message' => __('Username and password required.', 'smstunnel')));
        }

        // Verify credentials first
        $user = wp_authenticate($username, $password);
        if (is_wp_error($user)) {
            wp_send_json_error(array('message' => $user->get_error_message()));
        }

        // Check if bypass was already set
        $bypass = get_transient('smstunnel_2fa_bypass_' . $user->ID);
        if ($bypass === 'server_unavailable') {
            wp_send_json_success(array(
                'requires_2fa' => false,
                'bypass' => true,
                'authenticated' => true,
                'message' => __('SMS service unavailable - logging in without 2FA', 'smstunnel')
            ));
            return;
        }

        // Check if user role requires 2FA
        $required_roles = get_option('smstunnel_2fa_roles', array('administrator'));
        if (!is_array($required_roles)) {
            $required_roles = array('administrator');
        }

        $user_roles = $user->roles;
        $needs_2fa = false;
        foreach ($user_roles as $role) {
            if (in_array($role, $required_roles)) {
                $needs_2fa = true;
                break;
            }
        }

        if (!$needs_2fa) {
            wp_send_json_success(array('requires_2fa' => false));
            return;
        }

        // Get user phone
        $phone = get_user_meta($user->ID, 'billing_phone', true);
        if (empty($phone)) {
            $phone = get_user_meta($user->ID, 'phone', true);
        }
        if (empty($phone)) {
            $phone = get_user_meta($user->ID, 'smstunnel_phone', true);
        }

        // No phone - require phone setup
        if (empty($phone)) {
            // Store user ID in transient for phone setup
            $setup_token = wp_generate_password(32, false);
            set_transient('smstunnel_phone_setup_' . $setup_token, $user->ID, 10 * MINUTE_IN_SECONDS);

            wp_send_json_success(array(
                'requires_2fa' => false,
                'requires_phone_setup' => true,
                'setup_token' => $setup_token,
                'message' => __('Phone number required for 2FA. Please enter your phone number.', 'smstunnel')
            ));
        }

        // Format phone number to E.164
        $phone = $this->format_phone_number($phone);

        // Validate E.164 format
        if (!$this->is_valid_e164($phone)) {
            $fallback_enabled = get_option('smstunnel_2fa_server_fallback', 'yes') === 'yes';
            if ($fallback_enabled) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging
                    error_log(sprintf('[SMSTunnel 2FA Bypass] User: %s (ID: %d) - Invalid phone format', $user->user_login, $user->ID));
                }
                set_transient('smstunnel_2fa_bypass_' . $user->ID, 'server_unavailable', 5 * MINUTE_IN_SECONDS);
                wp_send_json_success(array(
                    'requires_2fa' => false,
                    'bypass' => true,
                    'message' => __('Invalid phone - 2FA bypassed', 'smstunnel')
                ));
            }
            wp_send_json_error(array('message' => __('Invalid phone number format.', 'smstunnel')));
        }

        // Generate prefix and code
        $prefix = sprintf('%02d', wp_rand(10, 99));
        $code_length = intval(get_option('smstunnel_2fa_length', 6));
        $max_code = pow(10, $code_length) - 1;
        $code = sprintf('%0' . $code_length . 'd', wp_rand(0, $max_code));
        $full_code = $prefix . '-' . $code;
        $expiry = get_option('smstunnel_2fa_expiry', 5);

        set_transient('smstunnel_wp_2fa_' . $user->ID, $code, $expiry * MINUTE_IN_SECONDS);
        set_transient('smstunnel_wp_2fa_prefix_' . $user->ID, $prefix, $expiry * MINUTE_IN_SECONDS);

        // Build message
        $message_template = get_option('smstunnel_2fa_message', __('Your {site_name} verification code is: {code}. Valid for {expiry} minutes.', 'smstunnel'));
        $message = str_replace(
            array('{code}', '{site_name}', '{expiry}'),
            array($full_code, get_bloginfo('name'), $expiry),
            $message_template
        );

        $result = smstunnel_send_sms($phone, $message, array(
            'is_2fa' => true,
            'source' => 'wp-login-2fa',
        ));

        if ($result['success']) {
            $masked_phone = substr($phone, 0, -4) . '****';
            wp_send_json_success(array(
                'requires_2fa' => true,
                'prefix' => $prefix,
                /* translators: %s: masked phone number */
                'message' => sprintf(__('Code sent to %s', 'smstunnel'), $masked_phone)
            ));
        } else {
            $fallback_enabled = get_option('smstunnel_2fa_server_fallback', 'yes') === 'yes';
            $error_msg = isset($result['error']) ? $result['error'] : __('Failed to send code.', 'smstunnel');

            if ($fallback_enabled) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging
                    error_log(sprintf('[SMSTunnel 2FA Bypass] User: %s (ID: %d) - SMS failed: %s', $user->user_login, $user->ID, $error_msg));
                }
                set_transient('smstunnel_2fa_bypass_' . $user->ID, 'server_unavailable', 5 * MINUTE_IN_SECONDS);
                wp_send_json_success(array(
                    'requires_2fa' => false,
                    'bypass' => true,
                    'message' => __('SMS unavailable - 2FA bypassed', 'smstunnel')
                ));
            } else {
                wp_send_json_error(array('message' => $error_msg));
            }
        }
    }

    /**
     * AJAX: Verify phone number and send confirmation code for phone setup
     */
    public function ajax_verify_phone_setup() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called during login flow without nonce
        $setup_token = isset($_POST['setup_token']) ? sanitize_text_field(wp_unslash($_POST['setup_token'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called during login flow without nonce
        $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';

        // Validate setup token
        $user_id = get_transient('smstunnel_phone_setup_' . $setup_token);
        if (!$user_id) {
            wp_send_json_error(array('message' => __('Session expired. Please try logging in again.', 'smstunnel')));
        }

        // Format and validate phone
        $phone = $this->format_phone_number($phone);
        if (!$this->is_valid_e164($phone)) {
            wp_send_json_error(array('message' => __('Invalid phone number format. Please use international format (e.g., +40712345678).', 'smstunnel')));
        }

        // Check for premium-rate numbers
        if ($this->is_premium_number($phone)) {
            wp_send_json_error(array('message' => __('Premium-rate numbers are not allowed. Please use a standard mobile number.', 'smstunnel')));
        }

        // Generate confirmation code
        $code_length = intval(get_option('smstunnel_2fa_length', 6));
        $max_code = pow(10, $code_length) - 1;
        $code = sprintf('%0' . $code_length . 'd', wp_rand(0, $max_code));
        $expiry = get_option('smstunnel_2fa_expiry', 5);

        // Store code and phone for verification
        set_transient('smstunnel_phone_verify_code_' . $setup_token, $code, $expiry * MINUTE_IN_SECONDS);
        set_transient('smstunnel_phone_verify_number_' . $setup_token, $phone, $expiry * MINUTE_IN_SECONDS);

        // Send SMS
        /* translators: %1$s: site name, %2$s: verification code, %3$d: minutes */
        $message = sprintf(__('Your %1$s phone verification code is: %2$s. Valid for %3$d minutes.', 'smstunnel'),
            get_bloginfo('name'), $code, $expiry);

        $result = smstunnel_send_sms($phone, $message, array(
            'is_2fa' => true,
            'source' => 'phone-setup-verification',
        ));

        if ($result['success']) {
            $masked_phone = substr($phone, 0, -4) . '****';
            wp_send_json_success(array(
                /* translators: %s: masked phone number */
                'message' => sprintf(__('Verification code sent to %s', 'smstunnel'), $masked_phone)
            ));
        } else {
            wp_send_json_error(array('message' => $result['error'] ?? __('Failed to send verification code.', 'smstunnel')));
        }
    }

    /**
     * AJAX: Confirm phone verification code and complete setup
     */
    public function ajax_confirm_phone_setup() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called during login flow without nonce
        $setup_token = isset($_POST['setup_token']) ? sanitize_text_field(wp_unslash($_POST['setup_token'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called during login flow without nonce
        $code = isset($_POST['code']) ? sanitize_text_field(wp_unslash($_POST['code'])) : '';

        // Validate setup token
        $user_id = get_transient('smstunnel_phone_setup_' . $setup_token);
        if (!$user_id) {
            wp_send_json_error(array('message' => __('Session expired. Please try logging in again.', 'smstunnel')));
        }

        // Get stored code and phone
        $stored_code = get_transient('smstunnel_phone_verify_code_' . $setup_token);
        $phone = get_transient('smstunnel_phone_verify_number_' . $setup_token);

        if (!$stored_code || !$phone) {
            wp_send_json_error(array('message' => __('Verification code expired. Please request a new code.', 'smstunnel')));
        }

        if ($code !== $stored_code) {
            wp_send_json_error(array('message' => __('Invalid verification code.', 'smstunnel')));
        }

        // Save phone to user meta
        update_user_meta($user_id, 'smstunnel_phone', $phone);

        // Clean up transients
        delete_transient('smstunnel_phone_setup_' . $setup_token);
        delete_transient('smstunnel_phone_verify_code_' . $setup_token);
        delete_transient('smstunnel_phone_verify_number_' . $setup_token);

        // Set bypass to allow login (phone is now configured, next login will have normal 2FA)
        set_transient('smstunnel_2fa_bypass_' . $user_id, 'phone_setup_complete', 60);

        wp_send_json_success(array(
            'message' => __('Phone verified successfully! You can now log in.', 'smstunnel'),
            'phone_saved' => true
        ));
    }

    /**
     * AJAX: Skip phone setup and allow login without 2FA
     */
    public function ajax_skip_phone_setup() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Called during login flow without nonce
        $setup_token = isset($_POST['setup_token']) ? sanitize_text_field(wp_unslash($_POST['setup_token'])) : '';

        // Validate setup token
        $user_id = get_transient('smstunnel_phone_setup_' . $setup_token);
        if (!$user_id) {
            wp_send_json_error(array('message' => __('Session expired. Please try logging in again.', 'smstunnel')));
        }

        // Clean up setup transient
        delete_transient('smstunnel_phone_setup_' . $setup_token);

        // Set bypass to allow login without 2FA this time
        set_transient('smstunnel_2fa_bypass_' . $user_id, 'phone_setup_skipped', 60);

        wp_send_json_success(array(
            'message' => __('Phone setup skipped. Please add your phone number from your profile.', 'smstunnel'),
            'skipped' => true
        ));
    }

    /**
     * Get device fingerprint for remember device feature
     */
    private function get_device_fingerprint() {
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $ip = $this->get_client_ip();
        return md5($user_agent . $ip);
    }

    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip = '';
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_CLIENT_IP']));
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR'])))[0];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }
        return sanitize_text_field(trim($ip));
    }

    /**
     * Check if device is trusted
     */
    private function is_device_trusted($user_id) {
        if (get_option('smstunnel_2fa_remember_device', 'no') !== 'yes') {
            return false;
        }

        $fingerprint = $this->get_device_fingerprint();
        $devices = get_user_meta($user_id, 'smstunnel_trusted_devices', true);

        if (!is_array($devices) || !isset($devices[$fingerprint])) {
            return false;
        }

        $device = $devices[$fingerprint];

        // Check expiry
        if (!isset($device['expiry']) || $device['expiry'] < time()) {
            // Token expired, remove it
            unset($devices[$fingerprint]);
            update_user_meta($user_id, 'smstunnel_trusted_devices', $devices);
            return false;
        }

        // Check cookie token
        $cookie_name = 'smstunnel_device_' . md5(SECURE_AUTH_KEY . $user_id);
        $cookie_token = isset($_COOKIE[$cookie_name]) ? sanitize_text_field(wp_unslash($_COOKIE[$cookie_name])) : '';

        if (empty($cookie_token) || !wp_check_password($cookie_token, $device['token'])) {
            return false;
        }

        return true;
    }

    /**
     * Trust device for user
     */
    private function trust_device($user_id) {
        $fingerprint = $this->get_device_fingerprint();
        $token = wp_generate_password(64, false);
        $days = intval(get_option('smstunnel_2fa_remember_days', 30));
        $expiry = time() + ($days * DAY_IN_SECONDS);

        $devices = get_user_meta($user_id, 'smstunnel_trusted_devices', true);
        if (!is_array($devices)) {
            $devices = array();
        }

        $devices[$fingerprint] = array(
            'token' => wp_hash_password($token),
            'expiry' => $expiry,
            'created' => time(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            'ip' => $this->get_client_ip(),
        );

        update_user_meta($user_id, 'smstunnel_trusted_devices', $devices);

        // Set cookie
        $cookie_name = 'smstunnel_device_' . md5(SECURE_AUTH_KEY . $user_id);
        setcookie($cookie_name, $token, $expiry, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
    }

    /**
     * WordPress Login 2FA - Verify code on authenticate
     */
    public function wp_2fa_authenticate($user, $username, $password) {
        if (is_wp_error($user) || !($user instanceof WP_User)) {
            return $user;
        }

        // Skip for AJAX 2FA requests
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Checking action type only
        if (defined('DOING_AJAX') && DOING_AJAX && isset($_POST['action']) && $_POST['action'] === 'smstunnel_wp_send_2fa') {
            return $user;
        }

        // Check if 2FA was bypassed
        $bypass = get_transient('smstunnel_2fa_bypass_' . $user->ID);
        if ($bypass === 'server_unavailable' || $bypass === 'phone_setup_complete' || $bypass === 'phone_setup_skipped') {
            delete_transient('smstunnel_2fa_bypass_' . $user->ID);
            return $user;
        }

        // Check if user role requires 2FA
        $required_roles = get_option('smstunnel_2fa_roles', array('administrator'));
        if (!is_array($required_roles)) {
            $required_roles = array('administrator');
        }

        $user_roles = $user->roles;
        $needs_2fa = false;
        foreach ($user_roles as $role) {
            if (in_array($role, $required_roles)) {
                $needs_2fa = true;
                break;
            }
        }

        if (!$needs_2fa) {
            return $user;
        }

        // Check if device is trusted (remember device feature)
        if ($this->is_device_trusted($user->ID)) {
            return $user;
        }

        // Check for 2FA code
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WP login handles nonce
        $submitted_code = isset($_POST['smstunnel_2fa_code']) ? sanitize_text_field(wp_unslash($_POST['smstunnel_2fa_code'])) : '';
        $stored_code = get_transient('smstunnel_wp_2fa_' . $user->ID);

        if (!empty($submitted_code)) {
            if (!$stored_code) {
                return new WP_Error('2fa_expired', __('<strong>Error:</strong> Verification code expired. Please try again.', 'smstunnel'));
            }
            if ($submitted_code !== $stored_code) {
                return new WP_Error('2fa_invalid', __('<strong>Error:</strong> Invalid verification code.', 'smstunnel'));
            }

            // Success - clear code
            delete_transient('smstunnel_wp_2fa_' . $user->ID);
            delete_transient('smstunnel_wp_2fa_prefix_' . $user->ID);

            // Handle remember device if requested
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WP login handles nonce
            $remember_device = isset($_POST['smstunnel_remember_device']) && $_POST['smstunnel_remember_device'] === '1';
            if ($remember_device && get_option('smstunnel_2fa_remember_device', 'no') === 'yes') {
                $this->trust_device($user->ID);
            }

            return $user;
        }

        // No code submitted - send SMS
        if ($stored_code) {
            return new WP_Error('2fa_required', __('<strong>2FA Required:</strong> Please enter the verification code sent to your phone.', 'smstunnel'));
        }

        // Get user phone
        $phone = get_user_meta($user->ID, 'billing_phone', true);
        if (empty($phone)) {
            $phone = get_user_meta($user->ID, 'phone', true);
        }
        if (empty($phone)) {
            $phone = get_user_meta($user->ID, 'smstunnel_phone', true);
        }

        if (empty($phone)) {
            // No automatic bypass - user must go through phone setup flow
            // JavaScript will intercept and show phone setup form
            return new WP_Error('2fa_no_phone', __('<strong>Error:</strong> No phone number configured. Please reload and complete phone setup.', 'smstunnel'));
        }

        $phone = $this->format_phone_number($phone);
        if (!$this->is_valid_e164($phone)) {
            return new WP_Error('2fa_invalid_phone', __('<strong>Error:</strong> Invalid phone number format. Please update your phone in your profile.', 'smstunnel'));
        }

        // Generate and send code
        $prefix = sprintf('%02d', wp_rand(10, 99));
        $code_length = intval(get_option('smstunnel_2fa_length', 6));
        $max_code = pow(10, $code_length) - 1;
        $code = sprintf('%0' . $code_length . 'd', wp_rand(0, $max_code));
        $full_code = $prefix . '-' . $code;
        $expiry = get_option('smstunnel_2fa_expiry', 5);

        $message_template = get_option('smstunnel_2fa_message', __('Your {site_name} verification code is: {code}. Valid for {expiry} minutes.', 'smstunnel'));
        $message = str_replace(
            array('{code}', '{site_name}', '{expiry}'),
            array($full_code, get_bloginfo('name'), $expiry),
            $message_template
        );

        $result = smstunnel_send_sms($phone, $message, array(
            'is_2fa' => true,
            'source' => 'wp-login-2fa-server',
        ));

        if ($result['success']) {
            set_transient('smstunnel_wp_2fa_' . $user->ID, $code, $expiry * MINUTE_IN_SECONDS);
            set_transient('smstunnel_wp_2fa_prefix_' . $user->ID, $prefix, $expiry * MINUTE_IN_SECONDS);
            $masked_phone = substr($phone, 0, -4) . '****';
            /* translators: %1$s: masked phone number, %2$s: code prefix */
            return new WP_Error('2fa_required', sprintf(__('<strong>2FA Required:</strong> Verification code sent to %1$s. Enter code with prefix <strong>%2$s</strong>.', 'smstunnel'), $masked_phone, $prefix));
        } else {
            $fallback_enabled = get_option('smstunnel_2fa_server_fallback', 'yes') === 'yes';
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';

            if ($fallback_enabled) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging
                    error_log(sprintf('[SMSTunnel 2FA Bypass] User: %s (ID: %d) - SMS failed: %s', $user->user_login, $user->ID, $error_msg));
                }
                return $user;
            }
            /* translators: %s: error message */
            return new WP_Error('2fa_sms_failed', sprintf(__('<strong>Error:</strong> Failed to send verification code: %s', 'smstunnel'), $error_msg));
        }
    }

    /**
     * Add phone field to user profile
     */
    public function add_phone_field_to_profile($user) {
        $phone = get_user_meta($user->ID, 'smstunnel_phone', true);
        ?>
        <h3><?php esc_html_e('SMSTunnel 2FA', 'smstunnel'); ?></h3>
        <table class="form-table">
            <tr>
                <th><label for="smstunnel_phone"><?php esc_html_e('Phone Number', 'smstunnel'); ?></label></th>
                <td>
                    <input type="text" name="smstunnel_phone" id="smstunnel_phone"
                           value="<?php echo esc_attr($phone); ?>" class="regular-text"
                           placeholder="+40712345678">
                    <p class="description">
                        <?php esc_html_e('Phone number in international format (e.g., +40712345678). Used for 2FA verification.', 'smstunnel'); ?>
                    </p>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Save phone field from user profile
     */
    public function save_phone_field($user_id) {
        if (!current_user_can('edit_user', $user_id)) {
            return false;
        }

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- User profile update handles nonce
        if (isset($_POST['smstunnel_phone'])) {
            $phone = sanitize_text_field(wp_unslash($_POST['smstunnel_phone']));
            update_user_meta($user_id, 'smstunnel_phone', $phone);
        }
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }

    /**
     * Format phone number to E.164
     */
    public function format_phone_number($phone) {
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        if (preg_match('/^\+[1-9]\d{6,14}$/', $phone)) {
            return $phone;
        }

        $phone = ltrim($phone, '+');

        // Get country
        $country = '';
        if (function_exists('WC') && WC()->countries) {
            $country = WC()->countries->get_base_country();
        } else {
            $locale = get_locale();
            $locale_parts = explode('_', $locale);
            if (count($locale_parts) >= 2) {
                $country = strtoupper($locale_parts[1]);
            }
        }

        if (empty($country)) {
            $country = 'RO';
        }

        $calling_codes = array(
            'RO' => '40', 'US' => '1', 'GB' => '44', 'DE' => '49', 'FR' => '33',
            'IT' => '39', 'ES' => '34', 'NL' => '31', 'BE' => '32', 'AT' => '43',
            'CH' => '41', 'PL' => '48', 'CZ' => '420', 'HU' => '36', 'BG' => '359',
            'MD' => '373', 'UA' => '380',
        );

        $code = isset($calling_codes[$country]) ? $calling_codes[$country] : '40';

        if (strpos($phone, $code) === 0) {
            $phone = '+' . $phone;
        } else if (substr($phone, 0, 1) === '0') {
            $phone = '+' . $code . substr($phone, 1);
        } else {
            $phone = '+' . $code . $phone;
        }

        if (!preg_match('/^\+[1-9]\d{6,14}$/', $phone)) {
            return '+' . preg_replace('/[^0-9]/', '', $phone);
        }

        return $phone;
    }

    /**
     * Validate E.164 phone number format
     */
    public function is_valid_e164($phone) {
        return preg_match('/^\+[1-9]\d{6,14}$/', $phone) === 1;
    }

    /**
     * Check if phone number is a premium-rate number
     * Blocks short codes and known premium prefixes
     */
    public function is_premium_number($phone) {
        // Must be E.164 format
        if (!preg_match('/^\+[1-9]\d+$/', $phone)) {
            return false;
        }

        // Too short - likely a short code (premium services)
        if (strlen($phone) < 10) {
            return true;
        }

        // Premium rate prefixes by country
        $premium_patterns = array(
            '/^\+40(90|80)[0-9]+$/',   // Romania: 090x, 080x
            '/^\+44(90|91|98)[0-9]+$/', // UK: 09xx
            '/^\+1(900)[0-9]+$/',       // US/Canada: 1-900
            '/^\+49(900|137|138)[0-9]+$/', // Germany: 0900, 0137, 0138
            '/^\+33(89)[0-9]+$/',       // France: 089x
            '/^\+39(89|166|899)[0-9]+$/', // Italy: 89x, 166, 899
            '/^\+34(80[3-7]|90[3-7])[0-9]+$/', // Spain: 803-807, 903-907
        );

        foreach ($premium_patterns as $pattern) {
            if (preg_match($pattern, $phone)) {
                return true;
            }
        }

        return false;
    }
}

// Initialize plugin
SMSTunnel::get_instance();

// Activation hook
register_activation_hook(__FILE__, function() {
    // Force create tables
    global $wpdb;
    $table_name = $wpdb->prefix . 'smstunnel_log';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        phone varchar(50) NOT NULL,
        message text NOT NULL,
        status varchar(20) NOT NULL DEFAULT 'pending',
        message_id varchar(100) DEFAULT NULL,
        error_message text DEFAULT NULL,
        is_2fa tinyint(1) NOT NULL DEFAULT 0,
        order_id bigint(20) DEFAULT NULL,
        source varchar(50) DEFAULT 'manual',
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY phone (phone),
        KEY status (status),
        KEY created_at (created_at)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
});

/**
 * Global function to send SMS
 *
 * @param string $phone Phone number in E.164 format
 * @param string $message SMS message
 * @param array $args Optional arguments: is_2fa, order_id, source
 * @return array Result with success/error
 */
function smstunnel_send_sms($phone, $message, $args = array()) {
    $defaults = array(
        'is_2fa' => false,
        'order_id' => null,
        'source' => 'function',
    );
    $args = wp_parse_args($args, $defaults);

    global $wpdb;
    $table_name = $wpdb->prefix . 'smstunnel_log';

    // Insert log entry
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
    $wpdb->insert($table_name, array(
        'phone' => $phone,
        'message' => $message,
        'status' => 'sending',
        'is_2fa' => $args['is_2fa'] ? 1 : 0,
        'order_id' => $args['order_id'],
        'source' => $args['source'],
    ));
    $log_id = $wpdb->insert_id;

    // Send via API - use dedicated 2FA endpoint for verification messages
    $api = new SMSTunnel_API();
    if ($args['is_2fa']) {
        // Use dedicated 2FA endpoint with server-side code validation
        $result = $api->send_2fa_sms($phone, $message);
    } else {
        // Regular SMS
        $result = $api->send_sms($phone, $message, false);
    }

    // Update log entry
    if ($result['success']) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
        $wpdb->update($table_name, array(
            'status' => 'sent',
            'message_id' => isset($result['messageId']) ? $result['messageId'] : null,
        ), array('id' => $log_id));
    } else {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
        $wpdb->update($table_name, array(
            'status' => 'failed',
            'error_message' => $result['error'],
        ), array('id' => $log_id));
    }

    do_action('smstunnel_sms_sent', $result, $phone, $message, $args);

    return $result;
}

// Action hook handler
add_action('smstunnel_send_sms', function($phone, $message, $args = array()) {
    smstunnel_send_sms($phone, $message, $args);
}, 10, 3);
