<?php
/**
 * SMSTunnel REST API Class
 *
 * Handles REST API endpoints for Quick Setup callbacks and status polling
 *
 * @package SMSTunnel
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_REST_API {

    /**
     * Namespace
     */
    const NAMESPACE = 'smstunnel/v1';

    /**
     * Instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    /**
     * Register REST routes
     */
    public function register_routes() {
        // Setup callback - called by SMSTunnel server after phone scans QR
        register_rest_route(self::NAMESPACE, '/setup-callback', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_setup_callback'),
            'permission_callback' => '__return_true', // Public endpoint, verified by site_token
        ));

        // Setup status - polled by WordPress admin page
        register_rest_route(self::NAMESPACE, '/setup-status', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_setup_status'),
            'permission_callback' => array($this, 'check_admin_permission'),
        ));

        // Reset setup - admin only
        register_rest_route(self::NAMESPACE, '/setup-reset', array(
            'methods' => 'POST',
            'callback' => array($this, 'reset_setup'),
            'permission_callback' => array($this, 'check_admin_permission'),
        ));

        // Get QR data - admin only
        register_rest_route(self::NAMESPACE, '/qr-data', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_qr_data'),
            'permission_callback' => array($this, 'check_admin_permission'),
        ));

        // Health check - for server to verify WordPress is reachable
        register_rest_route(self::NAMESPACE, '/health', array(
            'methods' => 'GET',
            'callback' => array($this, 'health_check'),
            'permission_callback' => '__return_true',
        ));
    }

    /**
     * Check admin permission
     */
    public function check_admin_permission() {
        return current_user_can('manage_options');
    }

    /**
     * Handle setup callback from SMSTunnel server
     *
     * This is called when the phone scans the QR code and the server
     * processes the quick setup request.
     */
    public function handle_setup_callback($request) {
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-quick-setup.php';
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-encryption.php';

        $params = $request->get_json_params();

        // Verify site token
        $site_token = isset($params['site_token']) ? sanitize_text_field($params['site_token']) : '';
        $stored_token = SMSTunnel_Quick_Setup::get_site_token();

        if (empty($site_token) || $site_token !== $stored_token) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'invalid_token',
                'message' => __('Invalid or expired setup token.', 'smstunnel'),
            ), 403);
        }

        // Verify required fields
        $api_key = isset($params['api_key']) ? sanitize_text_field($params['api_key']) : '';
        $device_name = isset($params['device_name']) ? sanitize_text_field($params['device_name']) : '';
        $server_public_key = isset($params['server_public_key']) ? $params['server_public_key'] : '';
        $status = isset($params['status']) ? sanitize_text_field($params['status']) : '';

        if ($status === 'error') {
            $error_message = isset($params['message']) ? sanitize_text_field($params['message']) : __('Unknown error', 'smstunnel');
            SMSTunnel_Quick_Setup::set_error($error_message);

            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'setup_error',
                'message' => $error_message,
            ), 400);
        }

        if (empty($api_key)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'missing_api_key',
                'message' => __('API key is required.', 'smstunnel'),
            ), 400);
        }

        // Save server's public key if provided
        if (!empty($server_public_key)) {
            SMSTunnel_Encryption::set_server_public_key($server_public_key);
        }

        // Complete setup
        $result = SMSTunnel_Quick_Setup::complete_setup($api_key, $device_name);

        if ($result) {
            return new WP_REST_Response(array(
                'success' => true,
                'message' => __('Setup completed successfully.', 'smstunnel'),
            ), 200);
        }

        return new WP_REST_Response(array(
            'success' => false,
            'error' => 'setup_failed',
            'message' => __('Failed to save setup data.', 'smstunnel'),
        ), 500);
    }

    /**
     * Get setup status (for polling)
     */
    public function get_setup_status($request) {
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-quick-setup.php';

        $info = SMSTunnel_Quick_Setup::get_status_info();

        return new WP_REST_Response($info, 200);
    }

    /**
     * Reset setup
     */
    public function reset_setup($request) {
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-quick-setup.php';

        SMSTunnel_Quick_Setup::reset_setup();

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Setup has been reset.', 'smstunnel'),
            'qr_data' => SMSTunnel_Quick_Setup::get_qr_content(),
        ), 200);
    }

    /**
     * Get QR data
     */
    public function get_qr_data($request) {
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-quick-setup.php';

        return new WP_REST_Response(array(
            'success' => true,
            'qr_content' => SMSTunnel_Quick_Setup::get_qr_content(),
            'qr_data' => SMSTunnel_Quick_Setup::get_qr_data(),
        ), 200);
    }

    /**
     * Health check
     */
    public function health_check($request) {
        return new WP_REST_Response(array(
            'status' => 'ok',
            'site_name' => get_bloginfo('name'),
            'site_url' => get_site_url(),
            'plugin_version' => SMSTUNNEL_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => phpversion(),
            'timestamp' => time(),
        ), 200);
    }
}
