<?php
/**
 * SMSTunnel Quick Setup Class
 *
 * Handles the simplified QR-based setup flow
 *
 * @package SMSTunnel
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_Quick_Setup {

    /**
     * Option names
     */
    const OPTION_SITE_TOKEN = 'smstunnel_site_token';
    const OPTION_SETUP_STATUS = 'smstunnel_setup_status';
    const OPTION_SETUP_DEVICE = 'smstunnel_setup_device';
    const OPTION_SETUP_TIMESTAMP = 'smstunnel_setup_timestamp';

    /**
     * Setup statuses
     */
    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_ERROR = 'error';

    /**
     * Server URL
     */
    const SERVER_URL = 'https://smstunnel.io';

    /**
     * Instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'), 11);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Initialize on plugin activation
     */
    public static function activate() {
        // Generate site token if not exists
        if (empty(get_option(self::OPTION_SITE_TOKEN))) {
            self::generate_site_token();
        }

        // Generate encryption keys
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-encryption.php';
        SMSTunnel_Encryption::ensure_keys_exist();

        // Set initial status
        if (empty(get_option(self::OPTION_SETUP_STATUS))) {
            update_option(self::OPTION_SETUP_STATUS, self::STATUS_PENDING);
        }
    }

    /**
     * Generate unique site token
     */
    public static function generate_site_token() {
        $token = bin2hex(random_bytes(32));
        update_option(self::OPTION_SITE_TOKEN, $token);
        update_option(self::OPTION_SETUP_TIMESTAMP, time());
        return $token;
    }

    /**
     * Get site token
     */
    public static function get_site_token() {
        return get_option(self::OPTION_SITE_TOKEN, '');
    }

    /**
     * Get setup status
     */
    public static function get_setup_status() {
        return get_option(self::OPTION_SETUP_STATUS, self::STATUS_PENDING);
    }

    /**
     * Set setup as completed
     */
    public static function complete_setup($api_key, $device_name = '') {
        // Save API key
        update_option('smstunnel_api_key', $api_key);

        // Update status
        update_option(self::OPTION_SETUP_STATUS, self::STATUS_COMPLETED);

        // Save device name
        if (!empty($device_name)) {
            update_option(self::OPTION_SETUP_DEVICE, $device_name);
        }

        return true;
    }

    /**
     * Set setup error
     */
    public static function set_error($message) {
        update_option(self::OPTION_SETUP_STATUS, self::STATUS_ERROR);
        update_option('smstunnel_setup_error', $message);
    }

    /**
     * Reset setup (for regenerating QR)
     */
    public static function reset_setup() {
        self::generate_site_token();
        update_option(self::OPTION_SETUP_STATUS, self::STATUS_PENDING);
        delete_option(self::OPTION_SETUP_DEVICE);
        delete_option('smstunnel_setup_error');
        delete_option('smstunnel_api_key');

        // Regenerate encryption keys
        require_once SMSTUNNEL_PLUGIN_DIR . 'includes/class-smstunnel-encryption.php';
        SMSTunnel_Encryption::delete_keys();
        SMSTunnel_Encryption::ensure_keys_exist();
    }

    /**
     * Get QR code data
     */
    public static function get_qr_data() {
        // Minimal data for QR code - public_key is fetched separately via REST API
        $data = array(
            's' => self::SERVER_URL,
            'a' => 'qs',
            't' => self::get_site_token(),
            'u' => get_site_url(),
            'e' => get_option('admin_email'), // Admin email for account creation
        );

        return $data;
    }

    /**
     * Get QR code content (base64 encoded JSON)
     */
    public static function get_qr_content() {
        $data = self::get_qr_data();
        return base64_encode(json_encode($data));
    }

    /**
     * Add admin menu page
     */
    public function add_menu_page() {
        // Add Quick Setup as first submenu item
        add_submenu_page(
            'smstunnel',
            __('Quick Setup', 'smstunnel'),
            __('Quick Setup', 'smstunnel'),
            'manage_options',
            'smstunnel-quick-setup',
            array($this, 'render_page'),
            5 // Position - after Settings
        );
    }

    /**
     * Enqueue scripts for quick setup page
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'smstunnel_page_smstunnel-quick-setup') {
            return;
        }

        wp_enqueue_style(
            'smstunnel-quick-setup',
            SMSTUNNEL_PLUGIN_URL . 'assets/css/quick-setup.css',
            array(),
            SMSTUNNEL_VERSION
        );

        // QR Code library
        wp_enqueue_script(
            'qrcode-js',
            SMSTUNNEL_PLUGIN_URL . 'assets/js/qrcode.min.js',
            array(),
            '1.0.0',
            true
        );

        wp_enqueue_script(
            'smstunnel-quick-setup',
            SMSTUNNEL_PLUGIN_URL . 'assets/js/quick-setup.js',
            array('jquery', 'qrcode-js'),
            SMSTUNNEL_VERSION,
            true
        );

        wp_localize_script('smstunnel-quick-setup', 'smstunnelQuickSetup', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'rest_url' => rest_url('smstunnel/v1/'),
            'nonce' => wp_create_nonce('wp_rest'), // Must use wp_rest for REST API
            'status' => self::get_setup_status(),
            'strings' => array(
                'waiting' => __('Waiting for QR scan...', 'smstunnel'),
                'completed' => __('Setup completed!', 'smstunnel'),
                'error' => __('Setup failed', 'smstunnel'),
                'connected' => __('Connected to', 'smstunnel'),
            ),
        ));
    }

    /**
     * Render the quick setup page
     */
    public function render_page() {
        // Check if already setup
        $status = self::get_setup_status();
        $device_name = get_option(self::OPTION_SETUP_DEVICE, '');
        $api_key = get_option('smstunnel_api_key', '');

        // Handle reset action
        if (isset($_GET['action']) && sanitize_text_field(wp_unslash($_GET['action'])) === 'reset' && isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'smstunnel_reset_setup')) {
            self::reset_setup();
            wp_safe_redirect(admin_url('admin.php?page=smstunnel-quick-setup'));
            exit;
        }

        // Get QR data
        $qr_content = self::get_qr_content();
        $qr_data = self::get_qr_data();

        ?>
        <div class="wrap smstunnel-quick-setup-wrap">
            <h1><?php esc_html_e('SMSTunnel Quick Setup', 'smstunnel'); ?></h1>

            <?php if ($status === self::STATUS_COMPLETED && !empty($api_key)): ?>
                <!-- Setup Completed View -->
                <div class="smstunnel-setup-completed">
                    <div class="smstunnel-success-icon">
                        <span class="dashicons dashicons-yes-alt"></span>
                    </div>
                    <h2><?php esc_html_e('Setup Complete!', 'smstunnel'); ?></h2>
                    <p class="smstunnel-device-info">
                        <?php if (!empty($device_name)): ?>
                            <?php
                            /* translators: %s: device name */
                            printf(esc_html__('Connected device: %s', 'smstunnel'), '<strong>' . esc_html($device_name) . '</strong>');
                        ?>
                        <?php else: ?>
                            <?php esc_html_e('Your site is connected to SMSTunnel.', 'smstunnel'); ?>
                        <?php endif; ?>
                    </p>
                    <div class="smstunnel-actions">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel-send')); ?>" class="button button-primary button-hero">
                            <?php esc_html_e('Send SMS', 'smstunnel'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel')); ?>" class="button button-secondary">
                            <?php esc_html_e('Settings', 'smstunnel'); ?>
                        </a>
                    </div>
                    <p class="smstunnel-reset-link">
                        <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=smstunnel-quick-setup&action=reset'), 'smstunnel_reset_setup')); ?>">
                            <?php esc_html_e('Reset and setup again', 'smstunnel'); ?>
                        </a>
                    </p>
                </div>

            <?php else: ?>
                <!-- Setup Pending View -->
                <div class="smstunnel-setup-card">
                    <div class="smstunnel-steps">
                        <h2><?php esc_html_e('Setup in 3 simple steps:', 'smstunnel'); ?></h2>

                        <div class="smstunnel-step">
                            <span class="step-number">1</span>
                            <div class="step-content">
                                <h3><?php esc_html_e('Download the SMSTunnel app', 'smstunnel'); ?></h3>
                                <div class="app-buttons">
                                    <a href="https://play.google.com/store/apps/details?id=com.smstunnel" target="_blank" class="app-button app-button-android">
                                        <span class="dashicons dashicons-smartphone"></span>
                                        <?php esc_html_e('Google Play', 'smstunnel'); ?>
                                    </a>
                                    <a href="https://apps.apple.com/app/smstunnel" target="_blank" class="app-button app-button-ios">
                                        <span class="dashicons dashicons-smartphone"></span>
                                        <?php esc_html_e('App Store', 'smstunnel'); ?>
                                    </a>
                                </div>
                            </div>
                        </div>

                        <div class="smstunnel-step">
                            <span class="step-number">2</span>
                            <div class="step-content">
                                <h3><?php esc_html_e('Scan this QR code', 'smstunnel'); ?></h3>
                                <div class="qr-container" id="qr-container">
                                    <div id="qrcode"></div>
                                    <p class="qr-hint"><?php esc_html_e('Open the SMSTunnel app and scan this code', 'smstunnel'); ?></p>
                                </div>
                                <div class="smstunnel-status" id="setup-status">
                                    <span class="status-icon spinner is-active"></span>
                                    <span class="status-text"><?php esc_html_e('Waiting for QR scan...', 'smstunnel'); ?></span>
                                </div>
                            </div>
                        </div>

                        <div class="smstunnel-step">
                            <span class="step-number">3</span>
                            <div class="step-content">
                                <h3><?php esc_html_e('Done! Start sending SMS', 'smstunnel'); ?></h3>
                                <p><?php esc_html_e('Once connected, you can send SMS directly from WordPress.', 'smstunnel'); ?></p>
                            </div>
                        </div>
                    </div>

                    <div class="smstunnel-advanced-link">
                        <p>
                            <?php esc_html_e('Already have an SMSTunnel account?', 'smstunnel'); ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=smstunnel')); ?>">
                                <?php esc_html_e('Advanced Setup', 'smstunnel'); ?> &rarr;
                            </a>
                        </p>
                    </div>
                </div>

                <!-- Hidden QR data for JavaScript -->
                <script type="text/javascript">
                    var smstunnelQRData = <?php echo json_encode($qr_content); ?>;
                </script>

            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Check if setup is needed
     */
    public static function needs_setup() {
        $api_key = get_option('smstunnel_api_key', '');
        return empty($api_key);
    }

    /**
     * Get setup info for status endpoint
     */
    public static function get_status_info() {
        $status = self::get_setup_status();

        $info = array(
            'status' => $status,
            'timestamp' => get_option(self::OPTION_SETUP_TIMESTAMP, 0),
        );

        if ($status === self::STATUS_COMPLETED) {
            $info['device_name'] = get_option(self::OPTION_SETUP_DEVICE, '');
            $info['connected'] = true;
        } elseif ($status === self::STATUS_ERROR) {
            $info['error'] = get_option('smstunnel_setup_error', __('Unknown error', 'smstunnel'));
        }

        return $info;
    }
}
