<?php
/**
 * SMSTunnel Encryption Class
 *
 * Handles RSA key generation and encryption/decryption for E2E messaging
 *
 * @package SMSTunnel
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_Encryption {

    /**
     * Option names for storing keys
     */
    const OPTION_PUBLIC_KEY = 'smstunnel_public_key';
    const OPTION_PRIVATE_KEY = 'smstunnel_private_key';
    const OPTION_SERVER_PUBLIC_KEY = 'smstunnel_server_public_key';

    /**
     * Generate a new RSA key pair
     *
     * @return array|WP_Error Array with 'public' and 'private' keys or WP_Error
     */
    public static function generate_keypair() {
        // Check if OpenSSL is available
        if (!function_exists('openssl_pkey_new')) {
            return new WP_Error('openssl_missing', __('OpenSSL extension is required for encryption.', 'smstunnel'));
        }

        $config = array(
            'digest_alg' => 'sha256',
            'private_key_bits' => 2048,
            'private_key_type' => OPENSSL_KEYTYPE_RSA,
        );

        // Generate key pair
        $res = openssl_pkey_new($config);

        if ($res === false) {
            return new WP_Error('key_generation_failed', __('Failed to generate RSA key pair.', 'smstunnel'));
        }

        // Extract private key
        openssl_pkey_export($res, $private_key);

        // Extract public key
        $key_details = openssl_pkey_get_details($res);
        $public_key = $key_details['key'];

        return array(
            'private' => $private_key,
            'public' => $public_key
        );
    }

    /**
     * Generate and store keys if they don't exist
     *
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public static function ensure_keys_exist() {
        // Check if keys already exist
        $public_key = get_option(self::OPTION_PUBLIC_KEY);
        $private_key = get_option(self::OPTION_PRIVATE_KEY);

        if (!empty($public_key) && !empty($private_key)) {
            return true; // Keys already exist
        }

        // Generate new key pair
        $keys = self::generate_keypair();

        if (is_wp_error($keys)) {
            return $keys;
        }

        // Store keys (private key encrypted with WordPress SALT)
        $encrypted_private = self::encrypt_with_salt($keys['private']);

        update_option(self::OPTION_PUBLIC_KEY, base64_encode($keys['public']));
        update_option(self::OPTION_PRIVATE_KEY, $encrypted_private);

        return true;
    }

    /**
     * Get the public key
     *
     * @return string|false Base64 encoded public key or false
     */
    public static function get_public_key() {
        return get_option(self::OPTION_PUBLIC_KEY, false);
    }

    /**
     * Get the private key (decrypted)
     *
     * @return string|false Private key PEM or false
     */
    public static function get_private_key() {
        $encrypted = get_option(self::OPTION_PRIVATE_KEY, false);

        if (empty($encrypted)) {
            return false;
        }

        return self::decrypt_with_salt($encrypted);
    }

    /**
     * Set the server's public key
     *
     * @param string $public_key Base64 encoded server public key
     * @return bool
     */
    public static function set_server_public_key($public_key) {
        return update_option(self::OPTION_SERVER_PUBLIC_KEY, $public_key);
    }

    /**
     * Get the server's public key
     *
     * @return string|false Base64 encoded server public key or false
     */
    public static function get_server_public_key() {
        return get_option(self::OPTION_SERVER_PUBLIC_KEY, false);
    }

    /**
     * Encrypt a message using the server's public key
     *
     * @param string $message Plain text message
     * @return string|WP_Error Base64 encoded encrypted message or WP_Error
     */
    public static function encrypt($message) {
        $server_public_key = self::get_server_public_key();

        if (empty($server_public_key)) {
            return new WP_Error('no_server_key', __('Server public key not available.', 'smstunnel'));
        }

        $public_key_pem = base64_decode($server_public_key);

        $encrypted = '';
        $success = openssl_public_encrypt($message, $encrypted, $public_key_pem);

        if (!$success) {
            return new WP_Error('encryption_failed', __('Failed to encrypt message.', 'smstunnel'));
        }

        return base64_encode($encrypted);
    }

    /**
     * Decrypt a message using our private key
     *
     * @param string $encrypted_message Base64 encoded encrypted message
     * @return string|WP_Error Decrypted message or WP_Error
     */
    public static function decrypt($encrypted_message) {
        $private_key = self::get_private_key();

        if (empty($private_key)) {
            return new WP_Error('no_private_key', __('Private key not available.', 'smstunnel'));
        }

        $encrypted_data = base64_decode($encrypted_message);

        $decrypted = '';
        $success = openssl_private_decrypt($encrypted_data, $decrypted, $private_key);

        if (!$success) {
            return new WP_Error('decryption_failed', __('Failed to decrypt message.', 'smstunnel'));
        }

        return $decrypted;
    }

    /**
     * Encrypt data using WordPress SALT
     *
     * @param string $data Data to encrypt
     * @return string Encrypted data (base64 encoded)
     */
    private static function encrypt_with_salt($data) {
        $key = self::get_encryption_key();
        $iv_length = openssl_cipher_iv_length('aes-256-cbc');
        $iv = openssl_random_pseudo_bytes($iv_length);

        $encrypted = openssl_encrypt($data, 'aes-256-cbc', $key, 0, $iv);

        // Prepend IV to encrypted data
        return base64_encode($iv . $encrypted);
    }

    /**
     * Decrypt data using WordPress SALT
     *
     * @param string $encrypted_data Base64 encoded encrypted data
     * @return string Decrypted data
     */
    private static function decrypt_with_salt($encrypted_data) {
        $key = self::get_encryption_key();
        $data = base64_decode($encrypted_data);

        $iv_length = openssl_cipher_iv_length('aes-256-cbc');
        $iv = substr($data, 0, $iv_length);
        $encrypted = substr($data, $iv_length);

        return openssl_decrypt($encrypted, 'aes-256-cbc', $key, 0, $iv);
    }

    /**
     * Get encryption key from WordPress SALT
     *
     * @return string 32-byte encryption key
     */
    private static function get_encryption_key() {
        $salt = defined('AUTH_KEY') ? AUTH_KEY : 'smstunnel-default-key';
        return hash('sha256', $salt, true);
    }

    /**
     * Get public key fingerprint (first 16 chars of SHA256 hash)
     *
     * @return string|false Fingerprint or false
     */
    public static function get_public_key_fingerprint() {
        $public_key = self::get_public_key();

        if (empty($public_key)) {
            return false;
        }

        return substr(hash('sha256', $public_key), 0, 16);
    }

    /**
     * Delete all stored keys (used when resetting setup)
     *
     * @return void
     */
    public static function delete_keys() {
        delete_option(self::OPTION_PUBLIC_KEY);
        delete_option(self::OPTION_PRIVATE_KEY);
        delete_option(self::OPTION_SERVER_PUBLIC_KEY);
    }
}
