<?php
/**
 * SMSTunnel API Class
 *
 * Handles communication with the SMSTunnel.io API
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_API {

    /**
     * API base URL
     */
    private $api_base;

    /**
     * API key
     */
    private $api_key;

    /**
     * Device ID (optional)
     */
    private $device_id;

    /**
     * Constructor
     */
    public function __construct() {
        // Get server URL from options, fallback to constant, then default
        $server_url = get_option('smstunnel_server_url', '');
        if (empty($server_url)) {
            $server_url = defined('SMSTUNNEL_API_BASE') ? SMSTUNNEL_API_BASE : 'https://smstunnel.io/api/v1';
        }
        $this->api_base = rtrim($server_url, '/');
        $this->api_key = get_option('smstunnel_api_key', '');
        $this->device_id = get_option('smstunnel_device_id', '');
    }

    /**
     * Send SMS
     *
     * @param string $to Phone number in E.164 format
     * @param string $message SMS message
     * @param bool $is2FA Whether this is a 2FA message
     * @return array Result with success/error
     */
    public function send_sms($to, $message, $is2FA = false) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        // Check if E2E encryption is enabled
        $e2e_key = get_option('smstunnel_e2e_key', array());

        if (!empty($e2e_key) && !empty($e2e_key['public_key']) && !empty($e2e_key['device_id'])) {
            // E2E Encryption enabled - encrypt the payload
            return $this->send_sms_encrypted($to, $message, $e2e_key['public_key'], $e2e_key['device_id'], $is2FA);
        }

        // Normal unencrypted send
        $body = array(
            'to'      => $to,
            'message' => $message,
            'is2FA'   => $is2FA,
        );

        if (!empty($this->device_id)) {
            $body['deviceId'] = $this->device_id;
        }

        // Use shorter timeout for 2FA requests (20 seconds)
        $timeout = $is2FA ? 20 : 30;
        $response = $this->request('POST', '/sms/send', $body, $timeout);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success'   => true,
                'messageId' => isset($body['data']['messageId']) ? $body['data']['messageId'] : null,
                'branded'   => isset($body['data']['branded']) ? $body['data']['branded'] : false,
                'remaining' => isset($body['data']['remaining']) ? $body['data']['remaining'] : null,
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Unknown error', 'smstunnel')
        );
    }

    /**
     * Send encrypted SMS (E2E)
     *
     * @param string $to Phone number
     * @param string $message Message text
     * @param string $public_key Device's RSA public key (base64)
     * @param string $device_id Device ID for encrypted messages
     * @param bool $is2FA Whether this is a 2FA message
     * @return array Result with success/error
     */
    private function send_sms_encrypted($to, $message, $public_key, $device_id, $is2FA = false) {
        // Create payload to encrypt
        $payload = json_encode(array(
            'to' => $to,
            'message' => $message
        ));

        // Encrypt with device's RSA public key
        $encrypted = '';
        $public_key_pem = "-----BEGIN PUBLIC KEY-----\n" .
                          chunk_split($public_key, 64, "\n") .
                          "-----END PUBLIC KEY-----";

        $key_resource = openssl_pkey_get_public($public_key_pem);
        if (!$key_resource) {
            return array(
                'success' => false,
                'error'   => __('Invalid device public key. Please re-scan the QR code.', 'smstunnel')
            );
        }

        // RSA encrypt with PKCS1 v1.5 padding (compatible with Flutter encrypt package)
        if (!openssl_public_encrypt($payload, $encrypted, $key_resource, OPENSSL_PKCS1_PADDING)) {
            return array(
                'success' => false,
                'error'   => __('Encryption failed. The message may be too long for RSA encryption.', 'smstunnel')
            );
        }

        // Send encrypted payload
        $body = array(
            'encrypted'        => true,
            'encryptedPayload' => base64_encode($encrypted),
            'deviceId'         => $device_id,
            'is2FA'            => $is2FA,
        );

        $timeout = $is2FA ? 20 : 30;
        $response = $this->request('POST', '/sms/send', $body, $timeout);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($response_body['success']) && $response_body['success']) {
            return array(
                'success'   => true,
                'messageId' => isset($response_body['data']['messageId']) ? $response_body['data']['messageId'] : null,
                'encrypted' => true,
                'remaining' => isset($response_body['data']['remaining']) ? $response_body['data']['remaining'] : null,
            );
        }

        return array(
            'success' => false,
            'error'   => isset($response_body['message']) ? $response_body['message'] : __('Failed to send encrypted SMS', 'smstunnel')
        );
    }

    /**
     * Send 2FA verification SMS
     *
     * Uses the dedicated 2FA endpoint with server-side code validation.
     * Server validates that message contains a valid 2FA code pattern.
     * Uses 2FA quota instead of regular SMS quota.
     *
     * @param string $to Phone number in E.164 format
     * @param string $message SMS message containing 2FA code
     * @return array Result with success/error
     */
    public function send_2fa_sms($to, $message) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        $body = array(
            'to'      => $to,
            'message' => $message,
        );

        if (!empty($this->device_id)) {
            $body['deviceId'] = $this->device_id;
        }

        // Use shorter timeout for 2FA requests (20 seconds)
        $response = $this->request('POST', '/sms/send-2fa', $body, 20);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success'   => true,
                'messageId' => isset($body['data']['messageId']) ? $body['data']['messageId'] : null,
                'branded'   => isset($body['data']['branded']) ? $body['data']['branded'] : false,
                'remaining' => isset($body['data']['remaining']) ? $body['data']['remaining'] : null,
            );
        }

        // Handle specific error for invalid 2FA code
        if ($code === 400) {
            return array(
                'success' => false,
                'error'   => isset($body['message']) ? $body['message'] : __('Invalid 2FA message format', 'smstunnel')
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Unknown error', 'smstunnel')
        );
    }

    /**
     * Send bulk SMS
     *
     * @param array $recipients Array of phone numbers
     * @param string $message SMS message
     * @return array Result with success/error
     */
    public function send_bulk_sms($recipients, $message) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        $body = array(
            'to'      => $recipients,
            'message' => $message,
        );

        if (!empty($this->device_id)) {
            $body['deviceId'] = $this->device_id;
        }

        $response = $this->request('POST', '/sms/send-bulk', $body);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success'    => true,
                'messageIds' => isset($body['data']['messageIds']) ? $body['data']['messageIds'] : null,
                'count'      => isset($body['data']['count']) ? $body['data']['count'] : 0,
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Unknown error', 'smstunnel')
        );
    }

    /**
     * Test API connection
     *
     * @return array Result with success/error and device info
     */
    public function test_connection() {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        $response = $this->request('GET', '/devices');

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            $devices = isset($body['data']['devices']) ? $body['data']['devices'] : array();
            $online = isset($body['data']['online']) ? $body['data']['online'] : 0;

            return array(
                'success' => true,
                'devices' => $devices,
                'online'  => $online,
                'total'   => count($devices),
            );
        }

        if ($code === 401) {
            return array(
                'success' => false,
                'error'   => __('Invalid API key.', 'smstunnel')
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Connection failed.', 'smstunnel')
        );
    }

    /**
     * Get usage statistics
     *
     * @return array Usage data
     */
    public function get_usage() {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        $response = $this->request('GET', '/account/usage');

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success' => true,
                'data'    => isset($body['data']) ? $body['data'] : array(),
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Failed to get usage.', 'smstunnel')
        );
    }

    /**
     * Get message status
     *
     * @param string $messageId Message ID
     * @return array Message status
     */
    public function get_message_status($messageId) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        $response = $this->request('GET', '/sms/status/' . $messageId);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success' => true,
                'data'    => isset($body['data']) ? $body['data'] : array(),
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Failed to get status.', 'smstunnel')
        );
    }

    /**
     * Get received SMS messages (inbox)
     *
     * @param int $limit Number of messages to retrieve (default 50, max 100)
     * @param int $page Page number for pagination
     * @param string $since ISO date string - only return messages after this date
     * @param string $phone Filter by sender phone number
     * @param string $device_id Filter by device ID
     * @return array Received messages
     */
    public function get_received_messages($limit = 50, $page = 1, $since = null, $phone = null, $device_id = null) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error'   => __('API key not configured.', 'smstunnel')
            );
        }

        // Build query params
        $query_params = array(
            'limit' => min($limit, 100),
            'page'  => max(1, intval($page)),
        );

        if (!empty($since)) {
            $query_params['since'] = $since;
        }

        if (!empty($phone)) {
            $query_params['phone'] = $phone;
        }

        if (!empty($device_id)) {
            $query_params['deviceId'] = $device_id;
        }

        $endpoint = '/sms/received?' . http_build_query($query_params);
        $response = $this->request('GET', $endpoint);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error'   => $response->get_error_message()
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && isset($body['success']) && $body['success']) {
            return array(
                'success'  => true,
                'messages' => isset($body['data']['messages']) ? $body['data']['messages'] : array(),
                'total'    => isset($body['data']['total']) ? $body['data']['total'] : 0,
                'page'     => isset($body['data']['page']) ? $body['data']['page'] : 1,
                'hasMore'  => isset($body['data']['hasMore']) ? $body['data']['hasMore'] : false,
            );
        }

        return array(
            'success' => false,
            'error'   => isset($body['message']) ? $body['message'] : __('Failed to get received messages.', 'smstunnel')
        );
    }

    /**
     * Make API request
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $body Request body
     * @param int $timeout Request timeout in seconds
     * @return array|WP_Error Response
     */
    private function request($method, $endpoint, $body = null, $timeout = 30) {
        $url = $this->api_base . $endpoint;

        $args = array(
            'method'  => $method,
            'headers' => array(
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $this->api_key,
            ),
            'timeout' => $timeout,
        );

        if ($body && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = json_encode($body);
        }

        return wp_remote_request($url, $args);
    }
}
