/**
 * SMSTunnel Social Login
 * Google, Facebook, and Email authentication
 */

(function($) {
    'use strict';

    var SocialLogin = {

        init: function() {
            this.bindEvents();
            this.checkCallback();
        },

        // Bind click events
        bindEvents: function() {
            var self = this;

            $(document).on('click', '#smstunnel-google-login', function(e) {
                e.preventDefault();
                self.loginWithGoogle();
            });

            $(document).on('click', '#smstunnel-facebook-login', function(e) {
                e.preventDefault();
                self.loginWithFacebook();
            });

            $(document).on('click', '#smstunnel-email-login', function(e) {
                e.preventDefault();
                self.loginWithEmail();
            });

            // Allow Enter key on password field
            $(document).on('keypress', '#smstunnel-login-password', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    self.loginWithEmail();
                }
            });
        },

        // Check if we're returning from OAuth callback
        checkCallback: function() {
            var urlParams = new URLSearchParams(window.location.search);
            var authSuccess = urlParams.get('smstunnel_auth');
            var authError = urlParams.get('smstunnel_auth_error');

            if (authSuccess === 'success') {
                this.showSuccess('Connected successfully!');
                // Clean URL
                window.history.replaceState({}, document.title, window.location.pathname + '?page=smstunnel');
            } else if (authError) {
                this.showError(decodeURIComponent(authError));
                // Clean URL
                window.history.replaceState({}, document.title, window.location.pathname + '?page=smstunnel');
            }
        },

        // Google login - redirect to backend
        loginWithGoogle: function() {
            var callbackUrl = encodeURIComponent(smstunnelSocial.callbackUrl);
            var authUrl = 'https://smstunnel.io/auth/google?redirect=' + callbackUrl;
            window.location.href = authUrl;
        },

        // Facebook login - redirect to backend
        loginWithFacebook: function() {
            var callbackUrl = encodeURIComponent(smstunnelSocial.callbackUrl);
            var authUrl = 'https://smstunnel.io/auth/facebook?redirect=' + callbackUrl;
            window.location.href = authUrl;
        },

        // Email/Password login
        loginWithEmail: function() {
            var self = this;
            var email = $('#smstunnel-login-email').val();
            var password = $('#smstunnel-login-password').val();
            var $button = $('#smstunnel-email-login');

            if (!email || !password) {
                this.showError('Please enter email and password.');
                return;
            }

            // Disable button
            $button.prop('disabled', true).text('Logging in...');

            $.ajax({
                url: smstunnelSocial.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'smstunnel_email_login',
                    nonce: smstunnelSocial.nonce,
                    email: email,
                    password: password
                },
                success: function(response) {
                    if (response.success) {
                        self.showSuccess('Connected successfully!');
                        // Reload page to show connected state
                        setTimeout(function() {
                            window.location.reload();
                        }, 1000);
                    } else {
                        self.showError(response.data.message || 'Login failed.');
                        $button.prop('disabled', false).text('Login');
                    }
                },
                error: function() {
                    self.showError('Connection error. Please try again.');
                    $button.prop('disabled', false).text('Login');
                }
            });
        },

        // Show success message
        showSuccess: function(message) {
            this.showNotice(message, 'success');
        },

        // Show error message
        showError: function(message) {
            this.showNotice(message, 'error');
        },

        // Show notice
        showNotice: function(message, type) {
            var $container = $('.smstunnel-connection-options');
            if (!$container.length) {
                $container = $('.smstunnel-connected');
            }
            var noticeClass = type === 'success' ? 'notice-success' : 'notice-error';

            // Remove existing notices
            $('.smstunnel-auth-notice').remove();

            // Add new notice before options
            var $notice = $('<div class="notice ' + noticeClass + ' is-dismissible smstunnel-auth-notice" style="margin-bottom: 15px;"><p>' + message + '</p></div>');
            $container.before($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        SocialLogin.init();
    });

})(jQuery);
