/**
 * SMSTunnel Quick Setup JavaScript
 *
 * Handles QR code generation and status polling
 */

(function($) {
    'use strict';

    var SMSTunnelQuickSetup = {
        pollInterval: null,
        pollDelay: 2000, // 2 seconds
        maxPollTime: 300000, // 5 minutes
        startTime: null,

        init: function() {
            this.startTime = Date.now();

            // Generate QR code
            this.generateQRCode();

            // Start polling if status is pending
            if (smstunnelQuickSetup.status === 'pending') {
                this.startPolling();
            }

            // Handle reset button clicks
            $(document).on('click', '.smstunnel-reset-btn', this.handleReset.bind(this));
        },

        generateQRCode: function() {
            var container = document.getElementById('qrcode');
            if (!container || typeof smstunnelQRData === 'undefined') {
                return;
            }

            // Clear existing QR code
            container.innerHTML = '';

            // Check if QRCode library is available
            if (typeof QRCode !== 'undefined') {
                new QRCode(container, {
                    text: smstunnelQRData,
                    width: 256,
                    height: 256,
                    colorDark: '#000000',
                    colorLight: '#ffffff',
                    correctLevel: QRCode.CorrectLevel.M
                });
            } else {
                // Fallback: use API to generate QR
                this.generateQRCodeFallback(container, smstunnelQRData);
            }
        },

        generateQRCodeFallback: function(container, data) {
            // Use Google Charts API as fallback
            var img = document.createElement('img');
            img.src = 'https://chart.googleapis.com/chart?cht=qr&chs=256x256&chl=' + encodeURIComponent(data);
            img.alt = 'QR Code';
            img.style.width = '256px';
            img.style.height = '256px';
            container.appendChild(img);
        },

        startPolling: function() {
            var self = this;

            // Clear any existing interval
            if (this.pollInterval) {
                clearInterval(this.pollInterval);
            }

            // Poll immediately
            this.checkStatus();

            // Set up interval
            this.pollInterval = setInterval(function() {
                // Check if we've exceeded max poll time
                if (Date.now() - self.startTime > self.maxPollTime) {
                    self.stopPolling();
                    self.showTimeout();
                    return;
                }

                self.checkStatus();
            }, this.pollDelay);
        },

        stopPolling: function() {
            if (this.pollInterval) {
                clearInterval(this.pollInterval);
                this.pollInterval = null;
            }
        },

        checkStatus: function() {
            var self = this;

            $.ajax({
                url: smstunnelQuickSetup.rest_url + 'setup-status',
                method: 'GET',
                beforeSend: function(xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', smstunnelQuickSetup.nonce);
                },
                success: function(response) {
                    if (response.status === 'completed') {
                        self.stopPolling();
                        self.showSuccess(response.device_name);
                    } else if (response.status === 'error') {
                        self.stopPolling();
                        self.showError(response.error || smstunnelQuickSetup.strings.error);
                    }
                    // If still pending, continue polling
                },
                error: function(xhr, status, error) {
                    console.error('Status check failed:', error);
                    // Continue polling despite errors
                }
            });
        },

        showSuccess: function(deviceName) {
            var $status = $('#setup-status');
            $status.removeClass('status-error').addClass('status-success');
            $status.html(
                '<span class="status-icon dashicons dashicons-yes-alt"></span>' +
                '<span class="status-text">' + smstunnelQuickSetup.strings.completed +
                (deviceName ? ' - ' + smstunnelQuickSetup.strings.connected + ' ' + deviceName : '') +
                '</span>'
            );

            // Reload page after 2 seconds to show completed view
            setTimeout(function() {
                window.location.reload();
            }, 2000);
        },

        showError: function(message) {
            var $status = $('#setup-status');
            $status.removeClass('status-success').addClass('status-error');
            $status.html(
                '<span class="status-icon dashicons dashicons-warning"></span>' +
                '<span class="status-text">' + message + '</span>'
            );
        },

        showTimeout: function() {
            var $status = $('#setup-status');
            $status.removeClass('status-success').addClass('status-error');
            $status.html(
                '<span class="status-icon dashicons dashicons-clock"></span>' +
                '<span class="status-text">QR code expired. <a href="#" class="smstunnel-reset-btn">Generate new code</a></span>'
            );
        },

        handleReset: function(e) {
            e.preventDefault();

            var self = this;
            var $btn = $(e.target);

            $btn.text('Resetting...');

            $.ajax({
                url: smstunnelQuickSetup.rest_url + 'setup-reset',
                method: 'POST',
                beforeSend: function(xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', smstunnelQuickSetup.nonce);
                },
                success: function(response) {
                    if (response.success) {
                        // Update QR data and regenerate
                        smstunnelQRData = response.qr_data;
                        self.generateQRCode();

                        // Reset status
                        var $status = $('#setup-status');
                        $status.removeClass('status-error status-success');
                        $status.html(
                            '<span class="status-icon spinner is-active"></span>' +
                            '<span class="status-text">' + smstunnelQuickSetup.strings.waiting + '</span>'
                        );

                        // Restart polling
                        self.startTime = Date.now();
                        self.startPolling();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Reset failed:', error);
                    $btn.text('Try again');
                }
            });
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        SMSTunnelQuickSetup.init();
    });

})(jQuery);
