<?php
/**
 * Plugin Name: SMSTunnel Ninja Forms
 * Plugin URI: https://smstunnel.io/integrations/ninja-forms
 * Description: Send SMS notifications via SMSTunnel when Ninja Forms are submitted. Adds SMS actions to Ninja Forms.
 * Version: 1.0.0
 * Author: SMSTunnel.io
 * Author URI: https://smstunnel.io
 * Text Domain: smstunnel-ninja-forms
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Requires Plugins: ninja-forms
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @author Narcis Bodea <contact@narboweb.ro>
 * @copyright 2024-2025 NARBOWEB SRL
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SMSTUNNEL_NF_VERSION', '1.0.0');
define('SMSTUNNEL_NF_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMSTUNNEL_NF_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SMSTUNNEL_NF_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main SMSTunnel Ninja Forms Plugin Class
 */
class SMSTunnel_Ninja_Forms {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        add_action('admin_notices', array($this, 'check_dependencies'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check dependencies
        if (!$this->dependencies_met()) {
            return;
        }

        // Register actions
        add_filter('ninja_forms_register_actions', array($this, 'register_actions'));

        // Admin assets
        add_action('admin_enqueue_scripts', array($this, 'admin_assets'));
    }

    /**
     * Check if dependencies are met
     */
    private function dependencies_met() {
        return class_exists('Ninja_Forms');
    }

    /**
     * Display admin notices for missing dependencies
     */
    public function check_dependencies() {
        if (!class_exists('Ninja_Forms')) {
            echo '<div class="notice notice-error"><p>';
            echo wp_kses(
                __('<strong>SMSTunnel Ninja Forms</strong> requires <strong>Ninja Forms</strong> to be installed and activated.', 'smstunnel-ninja-forms'),
                array('strong' => array())
            );
            echo '</p></div>';
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            echo '<div class="notice notice-warning is-dismissible"><p>';
            echo wp_kses(
                sprintf(
                    /* translators: %s: URL to SMSTunnel plugin page */
                    __('<strong>SMSTunnel Ninja Forms</strong>: The SMSTunnel plugin is not active. SMS notifications will not be sent until you install and configure the <a href="%s" target="_blank">SMSTunnel</a> plugin.', 'smstunnel-ninja-forms'),
                    'https://smstunnel.io/wordpress'
                ),
                array(
                    'strong' => array(),
                    'a'      => array(
                        'href'   => array(),
                        'target' => array(),
                    ),
                )
            );
            echo '</p></div>';
        }
    }

    /**
     * Register actions
     */
    public function register_actions($actions) {
        require_once SMSTUNNEL_NF_PLUGIN_DIR . 'includes/class-sms-action.php';

        $actions['smstunnel_sms'] = new SMSTunnel_NF_SMS_Action();
        $actions['smstunnel_sms_2'] = new SMSTunnel_NF_SMS_2_Action();

        return $actions;
    }

    /**
     * Enqueue admin assets
     */
    public function admin_assets($hook) {
        if (false === strpos($hook, 'ninja-forms')) {
            return;
        }

        wp_enqueue_style(
            'smstunnel-nf-admin',
            SMSTUNNEL_NF_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SMSTUNNEL_NF_VERSION
        );
    }
}

// Initialize plugin
SMSTunnel_Ninja_Forms::get_instance();
