<?php
/**
 * SMSTunnel Ninja Forms - SMS Action Classes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SMS Action for Ninja Forms
 */
class SMSTunnel_NF_SMS_Action extends NF_Abstracts_Action {

    /**
     * @var string
     */
    protected $_name = 'smstunnel_sms';

    /**
     * @var array
     */
    protected $_tags = array();

    /**
     * @var string
     */
    protected $_timing = 'normal';

    /**
     * @var int
     */
    protected $_priority = 10;

    /**
     * @var string
     */
    protected $_icon = "phone";

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();

        $this->_nicename = __('SMSTunnel SMS', 'smstunnel-ninja-forms');
        $this->_image = SMSTUNNEL_NF_PLUGIN_URL . 'assets/images/icon.png';

        $this->_settings = array(
            'smstunnel_sms_recipient' => array(
                'name'        => 'smstunnel_sms_recipient',
                'type'        => 'textbox',
                'group'       => 'primary',
                'label'       => __('Recipient Phone Number(s)', 'smstunnel-ninja-forms'),
                'placeholder' => '+40712345678',
                'width'       => 'full',
                'use_merge_tags' => array(
                    'include' => array(
                        'calcs',
                        'fields',
                    ),
                ),
                'help'        => __('Enter phone number(s) in international format. Separate multiple numbers with commas. You can use merge tags like {field:phone}.', 'smstunnel-ninja-forms'),
            ),
            'smstunnel_sms_body' => array(
                'name'        => 'smstunnel_sms_body',
                'type'        => 'textarea',
                'group'       => 'primary',
                'label'       => __('Message Body', 'smstunnel-ninja-forms'),
                'placeholder' => '',
                'width'       => 'full',
                'use_merge_tags' => array(
                    'include' => array(
                        'calcs',
                        'fields',
                    ),
                ),
                'help'        => __('Use merge tags like {field:name} to include form data. Use {all_fields} for all fields. Maximum 1600 characters.', 'smstunnel-ninja-forms'),
                'value'       => __("New form submission:\n{all_fields}", 'smstunnel-ninja-forms'),
            ),
        );
    }

    /**
     * Process action
     */
    public function process($action_settings, $form_id, $data) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return $data;
        }

        $recipient = isset($action_settings['smstunnel_sms_recipient']) ? $action_settings['smstunnel_sms_recipient'] : '';
        $body = isset($action_settings['smstunnel_sms_body']) ? $action_settings['smstunnel_sms_body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return $data;
        }

        // Get form fields
        $fields = array();
        if (isset($data['fields'])) {
            foreach ($data['fields'] as $field) {
                if (isset($field['key']) && isset($field['value'])) {
                    $fields[$field['key']] = $field['value'];
                }
            }
        }

        // Replace merge tags
        $recipient = $this->replace_merge_tags($recipient, $fields, $data);
        $body = $this->replace_merge_tags($body, $fields, $data);

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Parse recipients (comma-separated)
        $recipients = array_filter(array_map('trim', explode(',', $recipient)));

        // Send SMS to each recipient
        foreach ($recipients as $phone) {
            if (!empty($phone)) {
                smstunnel_send_sms($phone, $body, array(
                    'source' => 'ninja-forms'
                ));
            }
        }

        return $data;
    }

    /**
     * Replace merge tags in text
     */
    private function replace_merge_tags($text, $fields, $data) {
        // Replace {field:key}
        if (preg_match_all('/\{field:([^\}]+)\}/i', $text, $matches)) {
            foreach ($matches[1] as $index => $field_key) {
                $value = isset($fields[$field_key]) ? $fields[$field_key] : '';
                if (is_array($value)) {
                    $value = implode(', ', $value);
                }
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace {all_fields}
        if (strpos($text, '{all_fields}') !== false) {
            $all_fields = '';
            if (isset($data['fields'])) {
                foreach ($data['fields'] as $field) {
                    if (!empty($field['value'])) {
                        $value = is_array($field['value']) ? implode(', ', $field['value']) : $field['value'];
                        $label = isset($field['label']) ? $field['label'] : $field['key'];
                        $all_fields .= $label . ': ' . $value . "\n";
                    }
                }
            }
            $text = str_replace('{all_fields}', trim($all_fields), $text);
        }

        return $text;
    }
}

/**
 * SMS (2) Autoresponder Action for Ninja Forms
 */
class SMSTunnel_NF_SMS_2_Action extends NF_Abstracts_Action {

    /**
     * @var string
     */
    protected $_name = 'smstunnel_sms_2';

    /**
     * @var array
     */
    protected $_tags = array();

    /**
     * @var string
     */
    protected $_timing = 'normal';

    /**
     * @var int
     */
    protected $_priority = 10;

    /**
     * @var string
     */
    protected $_icon = "phone";

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();

        $this->_nicename = __('SMSTunnel SMS (2) Autoresponder', 'smstunnel-ninja-forms');
        $this->_image = SMSTUNNEL_NF_PLUGIN_URL . 'assets/images/icon.png';

        $this->_settings = array(
            'smstunnel_sms_2_description' => array(
                'name'  => 'smstunnel_sms_2_description',
                'type'  => 'html',
                'group' => 'primary',
                'label' => '',
                'width' => 'full',
                'value' => '<p>' . __('SMS (2) is used to send a confirmation message to the form submitter.', 'smstunnel-ninja-forms') . '</p>',
            ),
            'smstunnel_sms_2_recipient' => array(
                'name'        => 'smstunnel_sms_2_recipient',
                'type'        => 'textbox',
                'group'       => 'primary',
                'label'       => __('Recipient Phone Number', 'smstunnel-ninja-forms'),
                'placeholder' => '{field:phone}',
                'width'       => 'full',
                'use_merge_tags' => array(
                    'include' => array(
                        'calcs',
                        'fields',
                    ),
                ),
                'help'        => __('Use a merge tag like {field:phone} to get the phone number from a form field.', 'smstunnel-ninja-forms'),
            ),
            'smstunnel_sms_2_body' => array(
                'name'        => 'smstunnel_sms_2_body',
                'type'        => 'textarea',
                'group'       => 'primary',
                'label'       => __('Confirmation Message', 'smstunnel-ninja-forms'),
                'placeholder' => '',
                'width'       => 'full',
                'use_merge_tags' => array(
                    'include' => array(
                        'calcs',
                        'fields',
                    ),
                ),
                'help'        => __('Confirmation message sent to the form submitter. Maximum 1600 characters.', 'smstunnel-ninja-forms'),
                'value'       => __("Thank you for contacting us!\nWe have received your message and will get back to you soon.", 'smstunnel-ninja-forms'),
            ),
        );
    }

    /**
     * Process action
     */
    public function process($action_settings, $form_id, $data) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return $data;
        }

        $recipient = isset($action_settings['smstunnel_sms_2_recipient']) ? $action_settings['smstunnel_sms_2_recipient'] : '';
        $body = isset($action_settings['smstunnel_sms_2_body']) ? $action_settings['smstunnel_sms_2_body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return $data;
        }

        // Get form fields
        $fields = array();
        if (isset($data['fields'])) {
            foreach ($data['fields'] as $field) {
                if (isset($field['key']) && isset($field['value'])) {
                    $fields[$field['key']] = $field['value'];
                }
            }
        }

        // Replace merge tags
        $recipient = $this->replace_merge_tags($recipient, $fields, $data);
        $body = $this->replace_merge_tags($body, $fields, $data);

        // Skip if recipient is empty after replacement
        if (empty(trim($recipient))) {
            return $data;
        }

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Send SMS
        smstunnel_send_sms($recipient, $body, array(
            'source' => 'ninja-forms-autoresponder'
        ));

        return $data;
    }

    /**
     * Replace merge tags in text
     */
    private function replace_merge_tags($text, $fields, $data) {
        // Replace {field:key}
        if (preg_match_all('/\{field:([^\}]+)\}/i', $text, $matches)) {
            foreach ($matches[1] as $index => $field_key) {
                $value = isset($fields[$field_key]) ? $fields[$field_key] : '';
                if (is_array($value)) {
                    $value = implode(', ', $value);
                }
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace {all_fields}
        if (strpos($text, '{all_fields}') !== false) {
            $all_fields = '';
            if (isset($data['fields'])) {
                foreach ($data['fields'] as $field) {
                    if (!empty($field['value'])) {
                        $value = is_array($field['value']) ? implode(', ', $field['value']) : $field['value'];
                        $label = isset($field['label']) ? $field['label'] : $field['key'];
                        $all_fields .= $label . ': ' . $value . "\n";
                    }
                }
            }
            $text = str_replace('{all_fields}', trim($all_fields), $text);
        }

        return $text;
    }
}
