<?php
/**
 * SMSTunnel Formidable Forms - SMS Action Classes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SMS Action for Formidable Forms
 */
class SMSTunnel_Frm_SMS_Action extends FrmFormAction {

    /**
     * Constructor
     */
    public function __construct() {
        $action_ops = array(
            'classes'  => 'frm_sms_icon frm_icon_font',
            'limit'    => 99,
            'active'   => true,
            'event'    => array('create', 'update'),
            'priority' => 20,
        );

        $this->FrmFormAction('smstunnel_sms', __('SMSTunnel SMS', 'smstunnel-formidable'), $action_ops);
    }

    /**
     * Get default values for action settings
     */
    public function get_defaults() {
        return array(
            'recipient' => '',
            'body'      => __("New form submission:\n[default-message]", 'smstunnel-formidable'),
        );
    }

    /**
     * Render action settings form
     */
    public function form($form_action, $args = array()) {
        $defaults = $this->get_defaults();
        $form_action->post_content = wp_parse_args($form_action->post_content, $defaults);

        $form = isset($args['form']) ? $args['form'] : null;
        ?>
        <p class="frm_has_shortcodes">
            <label for="smstunnel_sms_recipient">
                <?php esc_html_e('Recipient Phone Number(s)', 'smstunnel-formidable'); ?>
            </label>
            <input type="text" id="smstunnel_sms_recipient" name="<?php echo esc_attr($this->get_field_name('recipient')); ?>"
                   class="frm_not_email_to large-text"
                   value="<?php echo esc_attr($form_action->post_content['recipient']); ?>"
                   placeholder="+40712345678">
            <span class="howto">
                <?php esc_html_e('Enter phone number(s) in international format. Separate multiple numbers with commas. You can use shortcodes like [25] (field ID).', 'smstunnel-formidable'); ?>
            </span>
        </p>

        <p class="frm_has_shortcodes">
            <label for="smstunnel_sms_body">
                <?php esc_html_e('Message Body', 'smstunnel-formidable'); ?>
            </label>
            <textarea id="smstunnel_sms_body" name="<?php echo esc_attr($this->get_field_name('body')); ?>"
                      class="frm_long_input" rows="6"><?php echo esc_textarea($form_action->post_content['body']); ?></textarea>
            <span class="howto">
                <?php esc_html_e('Use shortcodes to include form data. [default-message] shows all fields. Use [25] for specific field by ID. Maximum 1600 characters.', 'smstunnel-formidable'); ?>
            </span>
        </p>

        <div class="smstunnel-shortcodes-help" style="padding: 10px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px; margin-top: 15px;">
            <strong><?php esc_html_e('Available shortcodes:', 'smstunnel-formidable'); ?></strong><br>
            <code>[default-message]</code> - <?php esc_html_e('All submitted field values', 'smstunnel-formidable'); ?><br>
            <code>[id]</code> - <?php esc_html_e('Value of field with that ID', 'smstunnel-formidable'); ?><br>
            <code>[sitename]</code> - <?php esc_html_e('Website name', 'smstunnel-formidable'); ?><br>
            <code>[siteurl]</code> - <?php esc_html_e('Website URL', 'smstunnel-formidable'); ?><br>
            <code>[admin_email]</code> - <?php esc_html_e('Admin email', 'smstunnel-formidable'); ?>
            <?php if ($form): ?>
            <br><br><strong><?php esc_html_e('Form fields:', 'smstunnel-formidable'); ?></strong><br>
            <?php
            $fields = FrmField::get_all_for_form($form->id);
            if ($fields) {
                foreach ($fields as $field) {
                    if (in_array($field->type, array('divider', 'html', 'captcha', 'break'))) {
                        continue;
                    }
                    echo '<code>[' . esc_html($field->id) . ']</code> ' . esc_html($field->name) . '<br>';
                }
            }
            ?>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Trigger action on form submission
     */
    public static function trigger($action, $entry, $form) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $recipient = isset($action->post_content['recipient']) ? $action->post_content['recipient'] : '';
        $body = isset($action->post_content['body']) ? $action->post_content['body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return;
        }

        // Replace shortcodes
        $recipient = self::replace_shortcodes($recipient, $entry, $form);
        $body = self::replace_shortcodes($body, $entry, $form);

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Parse recipients (comma-separated)
        $recipients = array_filter(array_map('trim', explode(',', $recipient)));

        // Send SMS to each recipient
        foreach ($recipients as $phone) {
            if (!empty($phone)) {
                smstunnel_send_sms($phone, $body, array(
                    'source' => 'formidable-forms'
                ));
            }
        }
    }

    /**
     * Replace shortcodes in text
     */
    private static function replace_shortcodes($text, $entry, $form) {
        // Use Formidable's built-in shortcode processing
        if (class_exists('FrmEntryFormat')) {
            // Replace [default-message]
            if (strpos($text, '[default-message]') !== false) {
                $default_message = FrmEntryFormat::show_entry(array(
                    'id'          => $entry->id,
                    'plain_text'  => true,
                    'user_info'   => false,
                ));
                $text = str_replace('[default-message]', $default_message, $text);
            }
        }

        // Replace field shortcodes [id]
        if (preg_match_all('/\[(\d+)\]/', $text, $matches)) {
            foreach ($matches[1] as $index => $field_id) {
                $value = '';
                if (class_exists('FrmEntryMeta')) {
                    $value = FrmEntryMeta::get_entry_meta_by_field($entry->id, $field_id);
                    if (is_array($value)) {
                        $value = implode(', ', $value);
                    }
                }
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace site shortcodes
        $text = str_replace('[sitename]', get_bloginfo('name'), $text);
        $text = str_replace('[siteurl]', get_bloginfo('url'), $text);
        $text = str_replace('[admin_email]', get_option('admin_email'), $text);

        return $text;
    }
}

/**
 * SMS (2) Autoresponder Action for Formidable Forms
 */
class SMSTunnel_Frm_SMS_2_Action extends FrmFormAction {

    /**
     * Constructor
     */
    public function __construct() {
        $action_ops = array(
            'classes'  => 'frm_sms_icon frm_icon_font',
            'limit'    => 99,
            'active'   => true,
            'event'    => array('create'),
            'priority' => 25,
        );

        $this->FrmFormAction('smstunnel_sms_2', __('SMSTunnel SMS (2) Autoresponder', 'smstunnel-formidable'), $action_ops);
    }

    /**
     * Get default values for action settings
     */
    public function get_defaults() {
        return array(
            'recipient' => '',
            'body'      => __("Thank you for contacting us!\nWe have received your message and will get back to you soon.", 'smstunnel-formidable'),
        );
    }

    /**
     * Render action settings form
     */
    public function form($form_action, $args = array()) {
        $defaults = $this->get_defaults();
        $form_action->post_content = wp_parse_args($form_action->post_content, $defaults);

        $form = isset($args['form']) ? $args['form'] : null;
        ?>
        <p class="description" style="margin-bottom: 15px;">
            <?php esc_html_e('SMS (2) is used to send a confirmation message to the form submitter.', 'smstunnel-formidable'); ?>
        </p>

        <p class="frm_has_shortcodes">
            <label for="smstunnel_sms_2_recipient">
                <?php esc_html_e('Recipient Phone Number', 'smstunnel-formidable'); ?>
            </label>
            <input type="text" id="smstunnel_sms_2_recipient" name="<?php echo esc_attr($this->get_field_name('recipient')); ?>"
                   class="frm_not_email_to large-text"
                   value="<?php echo esc_attr($form_action->post_content['recipient']); ?>"
                   placeholder="[25]">
            <span class="howto">
                <?php esc_html_e('Use a shortcode like [25] to get the phone number from a form field (where 25 is your phone field ID).', 'smstunnel-formidable'); ?>
            </span>
        </p>

        <p class="frm_has_shortcodes">
            <label for="smstunnel_sms_2_body">
                <?php esc_html_e('Confirmation Message', 'smstunnel-formidable'); ?>
            </label>
            <textarea id="smstunnel_sms_2_body" name="<?php echo esc_attr($this->get_field_name('body')); ?>"
                      class="frm_long_input" rows="5"><?php echo esc_textarea($form_action->post_content['body']); ?></textarea>
            <span class="howto">
                <?php esc_html_e('Confirmation message sent to the form submitter. Maximum 1600 characters.', 'smstunnel-formidable'); ?>
            </span>
        </p>

        <?php if ($form): ?>
        <div class="smstunnel-shortcodes-help" style="padding: 10px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px; margin-top: 15px;">
            <strong><?php esc_html_e('Phone fields in this form:', 'smstunnel-formidable'); ?></strong><br>
            <?php
            $fields = FrmField::get_all_for_form($form->id);
            if ($fields) {
                foreach ($fields as $field) {
                    if ($field->type === 'phone' || strpos(strtolower($field->name), 'phone') !== false || strpos(strtolower($field->name), 'telefon') !== false) {
                        echo '<code>[' . esc_html($field->id) . ']</code> ' . esc_html($field->name) . '<br>';
                    }
                }
            }
            ?>
        </div>
        <?php endif; ?>
        <?php
    }

    /**
     * Trigger action on form submission
     */
    public static function trigger($action, $entry, $form) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $recipient = isset($action->post_content['recipient']) ? $action->post_content['recipient'] : '';
        $body = isset($action->post_content['body']) ? $action->post_content['body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return;
        }

        // Replace shortcodes
        $recipient = self::replace_shortcodes($recipient, $entry, $form);
        $body = self::replace_shortcodes($body, $entry, $form);

        // Skip if recipient is empty after replacement
        if (empty(trim($recipient))) {
            return;
        }

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Send SMS
        smstunnel_send_sms($recipient, $body, array(
            'source' => 'formidable-forms-autoresponder'
        ));
    }

    /**
     * Replace shortcodes in text
     */
    private static function replace_shortcodes($text, $entry, $form) {
        // Replace field shortcodes [id]
        if (preg_match_all('/\[(\d+)\]/', $text, $matches)) {
            foreach ($matches[1] as $index => $field_id) {
                $value = '';
                if (class_exists('FrmEntryMeta')) {
                    $value = FrmEntryMeta::get_entry_meta_by_field($entry->id, $field_id);
                    if (is_array($value)) {
                        $value = implode(', ', $value);
                    }
                }
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace site shortcodes
        $text = str_replace('[sitename]', get_bloginfo('name'), $text);
        $text = str_replace('[siteurl]', get_bloginfo('url'), $text);
        $text = str_replace('[admin_email]', get_option('admin_email'), $text);

        return $text;
    }
}

// Register action triggers
add_action('frm_trigger_smstunnel_sms_create_action', array('SMSTunnel_Frm_SMS_Action', 'trigger'), 10, 3);
add_action('frm_trigger_smstunnel_sms_update_action', array('SMSTunnel_Frm_SMS_Action', 'trigger'), 10, 3);
add_action('frm_trigger_smstunnel_sms_2_create_action', array('SMSTunnel_Frm_SMS_2_Action', 'trigger'), 10, 3);
