<?php
/**
 * Plugin Name: SMSTunnel for The Events Calendar
 * Plugin URI: https://smstunnel.io/integrations/the-events-calendar
 * Description: Send SMS notifications for The Events Calendar - RSVP confirmations, ticket purchases, event reminders, cancellations, and more.
 * Version: 1.0.0
 * Author: SMSTunnel.io
 * Author URI: https://smstunnel.io
 * Text Domain: smstunnel-events-calendar
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Requires Plugins: the-events-calendar
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @author Narcis Bodea <contact@narboweb.ro>
 * @copyright 2024-2025 NARBOWEB SRL
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SMSTUNNEL_TEC_VERSION', '1.0.0');
define('SMSTUNNEL_TEC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMSTUNNEL_TEC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SMSTUNNEL_TEC_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main SMSTunnel Events Calendar Plugin Class
 */
class SMSTunnel_Events_Calendar {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Plugin settings
     */
    private $settings = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'), 20);
        add_action('admin_notices', array($this, 'check_dependencies'));

        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check dependencies
        if (!$this->dependencies_met()) {
            return;
        }

        // Load includes
        $this->load_includes();

        // Admin assets
        add_action('admin_enqueue_scripts', array($this, 'admin_assets'));

        // Frontend assets
        add_action('wp_enqueue_scripts', array($this, 'frontend_assets'));
    }

    /**
     * Load include files
     */
    private function load_includes() {
        require_once SMSTUNNEL_TEC_PLUGIN_DIR . 'includes/class-settings.php';
        require_once SMSTUNNEL_TEC_PLUGIN_DIR . 'includes/class-phone-field.php';
        require_once SMSTUNNEL_TEC_PLUGIN_DIR . 'includes/class-sms-triggers.php';
        require_once SMSTUNNEL_TEC_PLUGIN_DIR . 'includes/class-reminders.php';
        require_once SMSTUNNEL_TEC_PLUGIN_DIR . 'includes/class-event-settings.php';

        // Initialize components
        $this->settings = new SMSTunnel_TEC_Settings();
        new SMSTunnel_TEC_Phone_Field();
        new SMSTunnel_TEC_SMS_Triggers();
        new SMSTunnel_TEC_Reminders();
        new SMSTunnel_TEC_Event_Settings();
    }

    /**
     * Check if dependencies are met
     */
    private function dependencies_met() {
        return class_exists('Tribe__Events__Main');
    }

    /**
     * Display admin notices for missing dependencies
     */
    public function check_dependencies() {
        if (!class_exists('Tribe__Events__Main')) {
            echo '<div class="notice notice-error"><p>';
            echo wp_kses(
                __('<strong>SMSTunnel for The Events Calendar</strong> requires <strong>The Events Calendar</strong> to be installed and activated.', 'smstunnel-events-calendar'),
                array('strong' => array())
            );
            echo '</p></div>';
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            echo '<div class="notice notice-warning is-dismissible"><p>';
            echo wp_kses(
                sprintf(
                    /* translators: %s: URL to SMSTunnel plugin page */
                    __('<strong>SMSTunnel for The Events Calendar</strong>: The SMSTunnel plugin is not active. SMS notifications will not be sent until you install and configure the <a href="%s" target="_blank">SMSTunnel</a> plugin.', 'smstunnel-events-calendar'),
                    'https://smstunnel.io/wordpress'
                ),
                array(
                    'strong' => array(),
                    'a'      => array(
                        'href'   => array(),
                        'target' => array(),
                    ),
                )
            );
            echo '</p></div>';
        }
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Schedule reminder cron job
        if (!wp_next_scheduled('smstunnel_tec_send_reminders')) {
            wp_schedule_event(time(), 'hourly', 'smstunnel_tec_send_reminders');
        }

        // Set default options
        $defaults = array(
            'enabled' => true,
            'rsvp_confirmation' => true,
            'ticket_confirmation' => true,
            'event_reminder' => true,
            'reminder_hours' => 24,
            'event_update' => true,
            'event_cancelled' => true,
            'organizer_notification' => false,
            'checkin_notification' => false,
            'rsvp_message' => __("Hi {attendee_name}!\n\nYou're registered for: {event_name}\nDate: {event_date}\nTime: {event_time}\nLocation: {event_location}\n\nSee you there!", 'smstunnel-events-calendar'),
            'ticket_message' => __("Hi {attendee_name}!\n\nYour ticket for {event_name} is confirmed!\nDate: {event_date}\nTime: {event_time}\nLocation: {event_location}\nTicket: {ticket_name}\n\nSee you there!", 'smstunnel-events-calendar'),
            'reminder_message' => __("Reminder: {event_name} is tomorrow!\nDate: {event_date}\nTime: {event_time}\nLocation: {event_location}\n\nSee you soon!", 'smstunnel-events-calendar'),
            'update_message' => __("Event Update: {event_name}\n\nThe event details have changed:\nDate: {event_date}\nTime: {event_time}\nLocation: {event_location}", 'smstunnel-events-calendar'),
            'cancelled_message' => __("Event Cancelled: {event_name}\n\nWe're sorry, but this event has been cancelled. We apologize for any inconvenience.", 'smstunnel-events-calendar'),
            'organizer_message' => __("New registration for {event_name}!\n\nAttendee: {attendee_name}\nEmail: {attendee_email}\nPhone: {attendee_phone}", 'smstunnel-events-calendar'),
        );

        $existing = get_option('smstunnel_tec_settings', array());
        if (empty($existing)) {
            update_option('smstunnel_tec_settings', $defaults);
        }
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled cron job
        $timestamp = wp_next_scheduled('smstunnel_tec_send_reminders');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'smstunnel_tec_send_reminders');
        }
    }

    /**
     * Get plugin settings
     */
    public static function get_settings() {
        return wp_parse_args(
            get_option('smstunnel_tec_settings', array()),
            array(
                'enabled' => true,
                'rsvp_confirmation' => true,
                'ticket_confirmation' => true,
                'event_reminder' => true,
                'reminder_hours' => 24,
                'event_update' => true,
                'event_cancelled' => true,
                'organizer_notification' => false,
                'checkin_notification' => false,
            )
        );
    }

    /**
     * Get a single setting
     */
    public static function get_setting($key, $default = null) {
        $settings = self::get_settings();
        return isset($settings[$key]) ? $settings[$key] : $default;
    }

    /**
     * Enqueue admin assets
     */
    public function admin_assets($hook) {
        // Load on TEC pages and our settings page
        if (strpos($hook, 'tribe') === false && strpos($hook, 'smstunnel') === false) {
            return;
        }

        wp_enqueue_style(
            'smstunnel-tec-admin',
            SMSTUNNEL_TEC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SMSTUNNEL_TEC_VERSION
        );
    }

    /**
     * Enqueue frontend assets
     */
    public function frontend_assets() {
        // Only load on event pages
        $is_event_query = function_exists('tribe_is_event_query') ? tribe_is_event_query() : false;
        if (!is_singular('tribe_events') && !$is_event_query) {
            return;
        }

        wp_enqueue_style(
            'smstunnel-tec-frontend',
            SMSTUNNEL_TEC_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            SMSTUNNEL_TEC_VERSION
        );
    }
}

// Initialize plugin
SMSTunnel_Events_Calendar::get_instance();
