<?php
/**
 * SMSTunnel Events Calendar - SMS Triggers
 *
 * Handles all SMS notification triggers
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_TEC_SMS_Triggers {

    /**
     * Constructor
     */
    public function __construct() {
        // RSVP created
        add_action('event_tickets_rsvp_attendee_created', array($this, 'on_rsvp_created'), 20, 4);

        // Ticket purchased (WooCommerce)
        add_action('event_tickets_woocommerce_attendee_created', array($this, 'on_ticket_created'), 20, 4);

        // Ticket purchased (EDD)
        add_action('event_tickets_edd_attendee_created', array($this, 'on_ticket_created'), 20, 4);

        // Event status change (update, cancelled)
        add_action('transition_post_status', array($this, 'on_event_status_change'), 10, 3);

        // Event meta update (date, time, location changes)
        add_action('tribe_events_update_meta', array($this, 'on_event_meta_update'), 10, 3);

        // Check-in
        add_action('rsvp_checkin', array($this, 'on_checkin'), 10, 2);
        add_action('event_tickets_checkin', array($this, 'on_checkin'), 10, 2);
    }

    /**
     * On RSVP created
     */
    public function on_rsvp_created($attendee_id, $event_id, $order_id, $product_id) {
        // Small delay to ensure phone is saved
        $this->schedule_immediate_sms('rsvp', $attendee_id, $event_id, $product_id);
    }

    /**
     * On ticket created
     */
    public function on_ticket_created($attendee_id, $event_id, $product_id, $order_id) {
        $this->schedule_immediate_sms('ticket', $attendee_id, $event_id, $product_id);
    }

    /**
     * Schedule immediate SMS (with small delay for data to be saved)
     */
    private function schedule_immediate_sms($type, $attendee_id, $event_id, $product_id) {
        // Process immediately - phone should be saved by now
        if ($type === 'rsvp') {
            $this->send_rsvp_confirmation($attendee_id, $event_id, $product_id);
        } else {
            $this->send_ticket_confirmation($attendee_id, $event_id, $product_id);
        }

        // Also send organizer notification
        $this->send_organizer_notification($attendee_id, $event_id);
    }

    /**
     * Send RSVP confirmation SMS
     */
    public function send_rsvp_confirmation($attendee_id, $event_id, $product_id = 0) {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['rsvp_confirmation']) {
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        // Check per-event settings
        $event_sms_enabled = get_post_meta($event_id, '_smstunnel_sms_enabled', true);
        if ($event_sms_enabled === '0') {
            return;
        }

        $phone = SMSTunnel_TEC_Phone_Field::get_attendee_phone($attendee_id);
        if (empty($phone)) {
            return;
        }

        // Get custom message or default
        $message = get_post_meta($event_id, '_smstunnel_rsvp_message', true);
        if (empty($message)) {
            $message = $settings['rsvp_message'];
        }

        // Replace placeholders
        $message = $this->replace_placeholders($message, $event_id, $attendee_id, $product_id);

        // Trim to max 1600 chars
        $message = mb_substr($message, 0, 1600);

        // Send SMS
        smstunnel_send_sms($phone, $message, array(
            'source' => 'the-events-calendar-rsvp'
        ));

        // Log
        $this->log_sms($event_id, $attendee_id, 'rsvp_confirmation', $phone);
    }

    /**
     * Send ticket purchase confirmation SMS
     */
    public function send_ticket_confirmation($attendee_id, $event_id, $product_id = 0) {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['ticket_confirmation']) {
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        // Check per-event settings
        $event_sms_enabled = get_post_meta($event_id, '_smstunnel_sms_enabled', true);
        if ($event_sms_enabled === '0') {
            return;
        }

        $phone = SMSTunnel_TEC_Phone_Field::get_attendee_phone($attendee_id);
        if (empty($phone)) {
            return;
        }

        // Get custom message or default
        $message = get_post_meta($event_id, '_smstunnel_ticket_message', true);
        if (empty($message)) {
            $message = $settings['ticket_message'];
        }

        // Replace placeholders
        $message = $this->replace_placeholders($message, $event_id, $attendee_id, $product_id);

        // Trim to max 1600 chars
        $message = mb_substr($message, 0, 1600);

        // Send SMS
        smstunnel_send_sms($phone, $message, array(
            'source' => 'the-events-calendar-ticket'
        ));

        // Log
        $this->log_sms($event_id, $attendee_id, 'ticket_confirmation', $phone);
    }

    /**
     * Send organizer notification
     */
    public function send_organizer_notification($attendee_id, $event_id) {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['organizer_notification']) {
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        // Get organizer phone (per-event or default)
        $organizer_phone = get_post_meta($event_id, '_smstunnel_organizer_phone', true);
        if (empty($organizer_phone)) {
            $organizer_phone = $settings['organizer_phone'] ?? '';
        }

        if (empty($organizer_phone)) {
            return;
        }

        // Get message
        $message = $settings['organizer_message'];

        // Replace placeholders
        $message = $this->replace_placeholders($message, $event_id, $attendee_id);

        // Trim to max 1600 chars
        $message = mb_substr($message, 0, 1600);

        // Send SMS
        smstunnel_send_sms($organizer_phone, $message, array(
            'source' => 'the-events-calendar-organizer'
        ));
    }

    /**
     * On event status change
     */
    public function on_event_status_change($new_status, $old_status, $post) {
        if ($post->post_type !== 'tribe_events') {
            return;
        }

        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled']) {
            return;
        }

        // Event cancelled (moved to trash)
        if ($new_status === 'trash' && $old_status === 'publish') {
            if ($settings['event_cancelled']) {
                $this->send_cancellation_sms($post->ID);
            }
        }
    }

    /**
     * Send cancellation SMS to all attendees
     */
    public function send_cancellation_sms($event_id) {
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $settings = SMSTunnel_Events_Calendar::get_settings();
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($event_id);

        if (empty($attendees)) {
            return;
        }

        // Get custom message or default
        $message = get_post_meta($event_id, '_smstunnel_cancelled_message', true);
        if (empty($message)) {
            $message = $settings['cancelled_message'];
        }

        foreach ($attendees as $attendee) {
            $attendee_message = $this->replace_placeholders($message, $event_id, $attendee['id']);
            $attendee_message = mb_substr($attendee_message, 0, 1600);

            smstunnel_send_sms($attendee['phone'], $attendee_message, array(
                'source' => 'the-events-calendar-cancelled'
            ));

            $this->log_sms($event_id, $attendee['id'], 'event_cancelled', $attendee['phone']);
        }
    }

    /**
     * On event meta update (date, time, location changes)
     */
    public function on_event_meta_update($event_id, $meta_key, $meta_value) {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['event_update']) {
            return;
        }

        // Track important meta changes
        $important_keys = array('_EventStartDate', '_EventEndDate', '_EventVenueID', '_EventAddress');

        if (!in_array($meta_key, $important_keys)) {
            return;
        }

        // Prevent duplicate notifications - use transient
        $transient_key = 'smstunnel_event_update_' . $event_id;
        if (get_transient($transient_key)) {
            return;
        }
        set_transient($transient_key, true, 60); // 1 minute debounce

        // Schedule update notification
        wp_schedule_single_event(time() + 30, 'smstunnel_tec_send_update_notification', array($event_id));
    }

    /**
     * Send update notification to all attendees
     */
    public static function send_update_notification($event_id) {
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $settings = SMSTunnel_Events_Calendar::get_settings();
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($event_id);

        if (empty($attendees)) {
            return;
        }

        // Get custom message or default
        $message = get_post_meta($event_id, '_smstunnel_update_message', true);
        if (empty($message)) {
            $message = $settings['update_message'];
        }

        $self = new self();
        foreach ($attendees as $attendee) {
            $attendee_message = $self->replace_placeholders($message, $event_id, $attendee['id']);
            $attendee_message = mb_substr($attendee_message, 0, 1600);

            smstunnel_send_sms($attendee['phone'], $attendee_message, array(
                'source' => 'the-events-calendar-update'
            ));

            $self->log_sms($event_id, $attendee['id'], 'event_update', $attendee['phone']);
        }
    }

    /**
     * On check-in
     */
    public function on_checkin($attendee_id, $qr_data = null) {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['checkin_notification']) {
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $phone = SMSTunnel_TEC_Phone_Field::get_attendee_phone($attendee_id);
        if (empty($phone)) {
            return;
        }

        // Get event ID from attendee
        $event_id = get_post_meta($attendee_id, '_tribe_tickets_event', true);
        if (!$event_id) {
            $event_id = get_post_meta($attendee_id, '_tribe_rsvp_event', true);
        }

        $message = __("You're checked in for {event_name}!\n\nEnjoy the event!", 'smstunnel-events-calendar');
        $message = $this->replace_placeholders($message, $event_id, $attendee_id);

        smstunnel_send_sms($phone, $message, array(
            'source' => 'the-events-calendar-checkin'
        ));
    }

    /**
     * Replace placeholders in message
     */
    public function replace_placeholders($message, $event_id, $attendee_id = 0, $product_id = 0) {
        // Get event data
        $event = get_post($event_id);
        $event_name = $event ? $event->post_title : '';

        // Get event date/time (with fallbacks if TEC functions don't exist)
        $event_date = function_exists('tribe_get_start_date')
            ? tribe_get_start_date($event_id, false, get_option('date_format'))
            : '';
        $event_time = function_exists('tribe_get_start_time')
            ? tribe_get_start_time($event_id, get_option('time_format'))
            : '';
        $event_end_time = function_exists('tribe_get_end_time')
            ? tribe_get_end_time($event_id, get_option('time_format'))
            : '';

        // Get venue/location
        $venue_id = function_exists('tribe_get_venue_id') ? tribe_get_venue_id($event_id) : 0;
        $venue_name = function_exists('tribe_get_venue') ? tribe_get_venue($event_id) : '';
        $venue_address = function_exists('tribe_get_full_address') ? tribe_get_full_address($event_id) : '';
        $event_location = $venue_name;
        if (!empty($venue_address)) {
            $event_location .= ', ' . $venue_address;
        }

        // Get event URL
        $event_url = get_permalink($event_id);

        // Get attendee data
        $attendee_name = '';
        $attendee_email = '';
        $attendee_phone = '';

        if ($attendee_id) {
            // Try to get from attendee meta
            $attendee_name = get_post_meta($attendee_id, '_tribe_tickets_full_name', true);
            if (empty($attendee_name)) {
                $attendee_name = get_post_meta($attendee_id, '_tribe_rsvp_full_name', true);
            }

            $attendee_email = get_post_meta($attendee_id, '_tribe_tickets_email', true);
            if (empty($attendee_email)) {
                $attendee_email = get_post_meta($attendee_id, '_tribe_rsvp_email', true);
            }

            $attendee_phone = SMSTunnel_TEC_Phone_Field::get_attendee_phone($attendee_id);
        }

        // Get ticket name
        $ticket_name = '';
        if ($product_id) {
            $ticket = get_post($product_id);
            $ticket_name = $ticket ? $ticket->post_title : '';
        }

        // Replace placeholders
        $replacements = array(
            '{event_name}'     => $event_name,
            '{event_date}'     => $event_date,
            '{event_time}'     => $event_time,
            '{event_end_time}' => $event_end_time,
            '{event_location}' => $event_location,
            '{event_venue}'    => $venue_name,
            '{event_address}'  => $venue_address,
            '{event_url}'      => $event_url,
            '{attendee_name}'  => $attendee_name,
            '{attendee_email}' => $attendee_email,
            '{attendee_phone}' => $attendee_phone,
            '{ticket_name}'    => $ticket_name,
            '{site_name}'      => get_bloginfo('name'),
            '{site_url}'       => get_bloginfo('url'),
        );

        return str_replace(array_keys($replacements), array_values($replacements), $message);
    }

    /**
     * Log SMS sent
     */
    private function log_sms($event_id, $attendee_id, $type, $phone) {
        $log = get_post_meta($event_id, '_smstunnel_sms_log', true);
        if (!is_array($log)) {
            $log = array();
        }

        $log[] = array(
            'attendee_id' => $attendee_id,
            'type'        => $type,
            'phone'       => $phone,
            'time'        => current_time('mysql'),
        );

        // Keep only last 100 entries
        if (count($log) > 100) {
            $log = array_slice($log, -100);
        }

        update_post_meta($event_id, '_smstunnel_sms_log', $log);
    }
}

// Hook for scheduled update notification
add_action('smstunnel_tec_send_update_notification', array('SMSTunnel_TEC_SMS_Triggers', 'send_update_notification'));
