<?php
/**
 * SMSTunnel Events Calendar - Global Settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_TEC_Settings {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_settings_page'), 20);
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_submenu_page(
            'edit.php?post_type=tribe_events',
            __('SMS Settings', 'smstunnel-events-calendar'),
            __('SMS Settings', 'smstunnel-events-calendar'),
            'manage_options',
            'smstunnel-tec-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('smstunnel_tec_settings', 'smstunnel_tec_settings', array($this, 'sanitize_settings'));
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        // Checkboxes
        $checkboxes = array(
            'enabled', 'rsvp_confirmation', 'ticket_confirmation',
            'event_reminder', 'event_update', 'event_cancelled',
            'organizer_notification', 'checkin_notification'
        );
        foreach ($checkboxes as $key) {
            $sanitized[$key] = isset($input[$key]) ? true : false;
        }

        // Numbers
        $sanitized['reminder_hours'] = isset($input['reminder_hours']) ? absint($input['reminder_hours']) : 24;

        // Text areas (messages)
        $textareas = array(
            'rsvp_message', 'ticket_message', 'reminder_message',
            'update_message', 'cancelled_message', 'organizer_message'
        );
        foreach ($textareas as $key) {
            $sanitized[$key] = isset($input[$key]) ? sanitize_textarea_field($input[$key]) : '';
        }

        // Phone number
        $sanitized['organizer_phone'] = isset($input['organizer_phone']) ? sanitize_text_field($input['organizer_phone']) : '';

        return $sanitized;
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        $settings = SMSTunnel_Events_Calendar::get_settings();
        $smstunnel_available = function_exists('smstunnel_send_sms');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('SMSTunnel SMS Settings', 'smstunnel-events-calendar'); ?></h1>

            <?php if (!$smstunnel_available): ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php esc_html_e('SMSTunnel plugin is not active.', 'smstunnel-events-calendar'); ?></strong>
                    <?php esc_html_e('Install and configure the SMSTunnel plugin to enable SMS notifications.', 'smstunnel-events-calendar'); ?>
                </p>
            </div>
            <?php endif; ?>

            <form method="post" action="options.php">
                <?php settings_fields('smstunnel_tec_settings'); ?>

                <!-- General Settings -->
                <div class="smstunnel-settings-section">
                    <h2><?php esc_html_e('General Settings', 'smstunnel-events-calendar'); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable SMS Notifications', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[enabled]" value="1" <?php checked($settings['enabled'], true); ?>>
                                    <?php esc_html_e('Enable SMS notifications for events', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Notification Types -->
                <div class="smstunnel-settings-section">
                    <h2><?php esc_html_e('Notification Types', 'smstunnel-events-calendar'); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('RSVP Confirmation', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[rsvp_confirmation]" value="1" <?php checked($settings['rsvp_confirmation'] ?? true, true); ?>>
                                    <?php esc_html_e('Send SMS when someone RSVPs to an event', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Ticket Purchase', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[ticket_confirmation]" value="1" <?php checked($settings['ticket_confirmation'] ?? true, true); ?>>
                                    <?php esc_html_e('Send SMS when someone purchases a ticket', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Event Reminder', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[event_reminder]" value="1" <?php checked($settings['event_reminder'] ?? true, true); ?>>
                                    <?php esc_html_e('Send reminder SMS before event starts', 'smstunnel-events-calendar'); ?>
                                </label>
                                <br><br>
                                <label>
                                    <?php esc_html_e('Send reminder', 'smstunnel-events-calendar'); ?>
                                    <input type="number" name="smstunnel_tec_settings[reminder_hours]" value="<?php echo esc_attr($settings['reminder_hours'] ?? 24); ?>" min="1" max="168" style="width: 60px;">
                                    <?php esc_html_e('hours before event', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Event Update', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[event_update]" value="1" <?php checked($settings['event_update'] ?? true, true); ?>>
                                    <?php esc_html_e('Send SMS when event details change (date, time, location)', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Event Cancelled', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[event_cancelled]" value="1" <?php checked($settings['event_cancelled'] ?? true, true); ?>>
                                    <?php esc_html_e('Send SMS when event is cancelled', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Organizer Notification', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[organizer_notification]" value="1" <?php checked($settings['organizer_notification'] ?? false, true); ?>>
                                    <?php esc_html_e('Send SMS to organizer when someone registers', 'smstunnel-events-calendar'); ?>
                                </label>
                                <br><br>
                                <label>
                                    <?php esc_html_e('Default organizer phone:', 'smstunnel-events-calendar'); ?>
                                    <input type="text" name="smstunnel_tec_settings[organizer_phone]" value="<?php echo esc_attr($settings['organizer_phone'] ?? ''); ?>" placeholder="+40712345678" class="regular-text">
                                </label>
                                <p class="description"><?php esc_html_e('Can be overridden per-event in event settings.', 'smstunnel-events-calendar'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Check-in Notification', 'smstunnel-events-calendar'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="smstunnel_tec_settings[checkin_notification]" value="1" <?php checked($settings['checkin_notification'] ?? false, true); ?>>
                                    <?php esc_html_e('Send SMS confirmation when attendee checks in', 'smstunnel-events-calendar'); ?>
                                </label>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Message Templates -->
                <div class="smstunnel-settings-section">
                    <h2><?php esc_html_e('Message Templates', 'smstunnel-events-calendar'); ?></h2>

                    <div class="smstunnel-placeholders-help">
                        <strong><?php esc_html_e('Available placeholders:', 'smstunnel-events-calendar'); ?></strong><br>
                        <code>{event_name}</code> - <?php esc_html_e('Event name', 'smstunnel-events-calendar'); ?><br>
                        <code>{event_date}</code> - <?php esc_html_e('Event date', 'smstunnel-events-calendar'); ?><br>
                        <code>{event_time}</code> - <?php esc_html_e('Event time', 'smstunnel-events-calendar'); ?><br>
                        <code>{event_location}</code> - <?php esc_html_e('Event location/venue', 'smstunnel-events-calendar'); ?><br>
                        <code>{event_url}</code> - <?php esc_html_e('Event URL', 'smstunnel-events-calendar'); ?><br>
                        <code>{attendee_name}</code> - <?php esc_html_e('Attendee name', 'smstunnel-events-calendar'); ?><br>
                        <code>{attendee_email}</code> - <?php esc_html_e('Attendee email', 'smstunnel-events-calendar'); ?><br>
                        <code>{attendee_phone}</code> - <?php esc_html_e('Attendee phone', 'smstunnel-events-calendar'); ?><br>
                        <code>{ticket_name}</code> - <?php esc_html_e('Ticket/RSVP type name', 'smstunnel-events-calendar'); ?><br>
                        <code>{site_name}</code> - <?php esc_html_e('Website name', 'smstunnel-events-calendar'); ?>
                    </div>

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="rsvp_message"><?php esc_html_e('RSVP Confirmation', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="rsvp_message" name="smstunnel_tec_settings[rsvp_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['rsvp_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="ticket_message"><?php esc_html_e('Ticket Purchase', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="ticket_message" name="smstunnel_tec_settings[ticket_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['ticket_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="reminder_message"><?php esc_html_e('Event Reminder', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="reminder_message" name="smstunnel_tec_settings[reminder_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['reminder_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="update_message"><?php esc_html_e('Event Update', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="update_message" name="smstunnel_tec_settings[update_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['update_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="cancelled_message"><?php esc_html_e('Event Cancelled', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="cancelled_message" name="smstunnel_tec_settings[cancelled_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['cancelled_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="organizer_message"><?php esc_html_e('Organizer Notification', 'smstunnel-events-calendar'); ?></label>
                            </th>
                            <td>
                                <textarea id="organizer_message" name="smstunnel_tec_settings[organizer_message]" rows="5" class="large-text code"><?php echo esc_textarea($settings['organizer_message'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}
