<?php
/**
 * SMSTunnel Events Calendar - Scheduled Reminders
 *
 * Handles scheduled event reminder SMS notifications
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_TEC_Reminders {

    /**
     * Meta key for tracking sent reminders
     */
    const REMINDER_SENT_META = '_smstunnel_reminder_sent';

    /**
     * Constructor
     */
    public function __construct() {
        // Hook into cron
        add_action('smstunnel_tec_send_reminders', array($this, 'process_reminders'));

        // Add custom cron interval if needed
        add_filter('cron_schedules', array($this, 'add_cron_interval'));
    }

    /**
     * Add custom cron interval
     */
    public function add_cron_interval($schedules) {
        $schedules['smstunnel_fifteen_minutes'] = array(
            'interval' => 900, // 15 minutes
            'display'  => __('Every 15 Minutes', 'smstunnel-events-calendar'),
        );
        return $schedules;
    }

    /**
     * Process reminders
     */
    public function process_reminders() {
        $settings = SMSTunnel_Events_Calendar::get_settings();

        if (!$settings['enabled'] || !$settings['event_reminder']) {
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $reminder_hours = isset($settings['reminder_hours']) ? absint($settings['reminder_hours']) : 24;

        // Find events happening in the next X hours
        $events = $this->get_upcoming_events($reminder_hours);

        foreach ($events as $event_id) {
            $this->send_event_reminders($event_id);
        }
    }

    /**
     * Get upcoming events that need reminders
     */
    private function get_upcoming_events($hours_before) {
        $now = current_time('timestamp');
        $reminder_window_start = $now;
        $reminder_window_end = $now + ($hours_before * 3600);

        // Query events starting within the reminder window
        $args = array(
            'post_type'      => 'tribe_events',
            'post_status'    => 'publish',
            'posts_per_page' => 50,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for event date filtering
            'meta_query'     => array(
                'relation' => 'AND',
                array(
                    'key'     => '_EventStartDate',
                    'value'   => gmdate('Y-m-d H:i:s', $reminder_window_start),
                    'compare' => '>=',
                    'type'    => 'DATETIME',
                ),
                array(
                    'key'     => '_EventStartDate',
                    'value'   => gmdate('Y-m-d H:i:s', $reminder_window_end),
                    'compare' => '<=',
                    'type'    => 'DATETIME',
                ),
            ),
        );

        $events = get_posts($args);
        $event_ids = array();

        foreach ($events as $event) {
            // Check if reminders already sent for this event
            $reminder_sent = get_post_meta($event->ID, self::REMINDER_SENT_META, true);

            if ($reminder_sent) {
                continue;
            }

            // Check per-event settings
            $event_reminders_enabled = get_post_meta($event->ID, '_smstunnel_reminder_enabled', true);
            if ($event_reminders_enabled === '0') {
                continue;
            }

            $event_ids[] = $event->ID;
        }

        return $event_ids;
    }

    /**
     * Send reminders for an event
     */
    private function send_event_reminders($event_id) {
        $settings = SMSTunnel_Events_Calendar::get_settings();
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($event_id);

        if (empty($attendees)) {
            // Mark as sent even if no attendees (to prevent repeated checks)
            update_post_meta($event_id, self::REMINDER_SENT_META, current_time('mysql'));
            return;
        }

        // Get custom message or default
        $message = get_post_meta($event_id, '_smstunnel_reminder_message', true);
        if (empty($message)) {
            $message = $settings['reminder_message'];
        }

        $triggers = new SMSTunnel_TEC_SMS_Triggers();
        $sent_count = 0;

        foreach ($attendees as $attendee) {
            // Check if reminder already sent to this attendee
            $attendee_reminder_sent = get_post_meta($attendee['id'], '_smstunnel_reminder_sent_' . $event_id, true);
            if ($attendee_reminder_sent) {
                continue;
            }

            $attendee_message = $triggers->replace_placeholders($message, $event_id, $attendee['id']);
            $attendee_message = mb_substr($attendee_message, 0, 1600);

            $result = smstunnel_send_sms($attendee['phone'], $attendee_message, array(
                'source' => 'the-events-calendar-reminder'
            ));

            if ($result) {
                // Mark as sent for this attendee
                update_post_meta($attendee['id'], '_smstunnel_reminder_sent_' . $event_id, current_time('mysql'));
                $sent_count++;
            }
        }

        // Mark event reminders as sent
        if ($sent_count > 0) {
            update_post_meta($event_id, self::REMINDER_SENT_META, current_time('mysql'));
            update_post_meta($event_id, '_smstunnel_reminder_count', $sent_count);
        }

        // Log (commented out for production)
        // phpcs:ignore Squiz.PHP.CommentedOutCode.Found
        // error_log(sprintf(
        //     'SMSTunnel TEC: Sent %d reminder(s) for event #%d',
        //     $sent_count,
        //     $event_id
        // ));
    }

    /**
     * Get reminder status for an event
     */
    public static function get_reminder_status($event_id) {
        $sent = get_post_meta($event_id, self::REMINDER_SENT_META, true);
        $count = get_post_meta($event_id, '_smstunnel_reminder_count', true);

        if ($sent) {
            return array(
                'sent'      => true,
                'sent_at'   => $sent,
                'count'     => $count ? absint($count) : 0,
            );
        }

        return array(
            'sent'  => false,
            'count' => 0,
        );
    }

    /**
     * Reset reminder status (useful for rescheduled events)
     */
    public static function reset_reminder_status($event_id) {
        delete_post_meta($event_id, self::REMINDER_SENT_META);
        delete_post_meta($event_id, '_smstunnel_reminder_count');

        // Also reset attendee-level tracking
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($event_id);
        foreach ($attendees as $attendee) {
            delete_post_meta($attendee['id'], '_smstunnel_reminder_sent_' . $event_id);
        }
    }

    /**
     * Manually trigger reminder for an event (admin action)
     */
    public static function send_manual_reminder($event_id) {
        // Reset first
        self::reset_reminder_status($event_id);

        // Send
        $instance = new self();
        $instance->send_event_reminders($event_id);

        return self::get_reminder_status($event_id);
    }
}
