<?php
/**
 * SMSTunnel Events Calendar - Phone Field Integration
 *
 * Adds phone number field to RSVP and ticket registration forms
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_TEC_Phone_Field {

    /**
     * Meta key for storing phone number
     */
    const PHONE_META_KEY = '_smstunnel_attendee_phone';

    /**
     * Constructor
     */
    public function __construct() {
        // Add phone field to RSVP form
        add_action('tribe_tickets_rsvp_form_extra_fields', array($this, 'add_rsvp_phone_field'), 10, 2);

        // Add phone field to ticket form (Event Tickets)
        add_action('tribe_tickets_ticket_email_form', array($this, 'add_ticket_phone_field'), 10, 2);

        // Alternative hook for ticket form
        add_action('tribe_tickets_before_front_end_ticket_form_submit', array($this, 'add_ticket_phone_field_alt'), 10, 1);

        // Save phone number on RSVP
        add_action('event_tickets_rsvp_attendee_created', array($this, 'save_rsvp_phone'), 10, 4);

        // Save phone number on ticket purchase (WooCommerce)
        add_action('event_tickets_woocommerce_attendee_created', array($this, 'save_woo_phone'), 10, 4);

        // Save phone number on ticket purchase (EDD)
        add_action('event_tickets_edd_attendee_created', array($this, 'save_edd_phone'), 10, 4);

        // Add phone to attendee table columns
        add_filter('tribe_tickets_attendee_table_columns', array($this, 'add_phone_column'));
        add_filter('tribe_events_tickets_attendees_table_column', array($this, 'render_phone_column'), 10, 3);

        // Make phone searchable in attendees
        add_filter('tribe_tickets_search_attendees_by', array($this, 'add_phone_search'));

        // Add phone to REST API attendee data
        add_filter('tribe_tickets_rest_api_attendee_data', array($this, 'add_phone_to_api'), 10, 2);
    }

    /**
     * Add phone field to RSVP form
     */
    public function add_rsvp_phone_field($event_id, $ticket) {
        $this->render_phone_field('rsvp');
    }

    /**
     * Add phone field to ticket form
     */
    public function add_ticket_phone_field($event_id, $ticket) {
        $this->render_phone_field('ticket');
    }

    /**
     * Alternative ticket phone field
     */
    public function add_ticket_phone_field_alt($event_id) {
        $this->render_phone_field('ticket');
    }

    /**
     * Render phone field HTML
     */
    private function render_phone_field($type = 'rsvp') {
        $field_name = 'smstunnel_phone';
        $field_id = 'smstunnel-phone-' . $type;

        // Get user phone if logged in
        $user_phone = '';
        if (is_user_logged_in()) {
            $user_phone = get_user_meta(get_current_user_id(), 'billing_phone', true);
            if (empty($user_phone)) {
                $user_phone = get_user_meta(get_current_user_id(), 'phone', true);
            }
        }
        ?>
        <tr class="tribe-tickets-phone-row smstunnel-phone-field">
            <td colspan="2">
                <label for="<?php echo esc_attr($field_id); ?>" class="tribe-tickets-phone-label">
                    <?php esc_html_e('Phone Number', 'smstunnel-events-calendar'); ?>
                    <span class="tribe-tickets-phone-required">*</span>
                </label>
                <input type="tel"
                       id="<?php echo esc_attr($field_id); ?>"
                       name="<?php echo esc_attr($field_name); ?>"
                       value="<?php echo esc_attr($user_phone); ?>"
                       placeholder="+40712345678"
                       class="tribe-tickets-phone-input"
                       required>
                <p class="tribe-tickets-phone-description">
                    <?php esc_html_e('Enter your phone number to receive SMS notifications about this event.', 'smstunnel-events-calendar'); ?>
                </p>
            </td>
        </tr>
        <?php
    }

    /**
     * Save phone number on RSVP creation
     */
    public function save_rsvp_phone($attendee_id, $post_id, $order_id, $product_id) {
        $this->save_phone($attendee_id);
    }

    /**
     * Save phone for WooCommerce ticket
     */
    public function save_woo_phone($attendee_id, $event_id, $product_id, $order_id) {
        $this->save_phone($attendee_id);

        // Also try to get from WooCommerce order
        if (class_exists('WC_Order') && $order_id) {
            $order = wc_get_order($order_id);
            if ($order) {
                $billing_phone = $order->get_billing_phone();
                if (!empty($billing_phone)) {
                    update_post_meta($attendee_id, self::PHONE_META_KEY, sanitize_text_field($billing_phone));
                }
            }
        }
    }

    /**
     * Save phone for EDD ticket
     */
    public function save_edd_phone($attendee_id, $event_id, $product_id, $order_id) {
        $this->save_phone($attendee_id);
    }

    /**
     * Generic save phone method
     */
    private function save_phone($attendee_id) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification is handled by the parent form submission (Event Tickets plugin).
        if (isset($_POST['smstunnel_phone']) && !empty($_POST['smstunnel_phone'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification is handled by the parent form submission.
            $phone = sanitize_text_field(wp_unslash($_POST['smstunnel_phone']));
            update_post_meta($attendee_id, self::PHONE_META_KEY, $phone);

            // Also save to user meta if logged in
            if (is_user_logged_in()) {
                update_user_meta(get_current_user_id(), 'phone', $phone);
            }
        }
    }

    /**
     * Get attendee phone number
     */
    public static function get_attendee_phone($attendee_id) {
        $phone = get_post_meta($attendee_id, self::PHONE_META_KEY, true);

        // Fallback: try to get from Event Tickets Plus meta
        if (empty($phone) && class_exists('Tribe__Tickets_Plus__Meta')) {
            $meta = get_post_meta($attendee_id, Tribe__Tickets_Plus__Meta::META_KEY, true);
            if (is_array($meta)) {
                // Look for phone field in custom meta
                foreach ($meta as $key => $value) {
                    if (stripos($key, 'phone') !== false || stripos($key, 'telefon') !== false) {
                        $phone = $value;
                        break;
                    }
                }
            }
        }

        // Fallback: try to get from attendee user
        if (empty($phone)) {
            $attendee = get_post($attendee_id);
            if ($attendee) {
                $user_id = get_post_meta($attendee_id, '_tribe_tickets_attendee_user_id', true);
                if ($user_id) {
                    $phone = get_user_meta($user_id, 'billing_phone', true);
                    if (empty($phone)) {
                        $phone = get_user_meta($user_id, 'phone', true);
                    }
                }
            }
        }

        return $phone;
    }

    /**
     * Add phone column to attendees table
     */
    public function add_phone_column($columns) {
        if (!is_array($columns)) {
            return $columns;
        }

        $new_columns = array();
        foreach ($columns as $key => $label) {
            $new_columns[$key] = $label;
            if ($key === 'primary_info') {
                $new_columns['phone'] = __('Phone', 'smstunnel-events-calendar');
            }
        }
        return $new_columns;
    }

    /**
     * Render phone column content
     */
    public function render_phone_column($value, $item, $column) {
        if ($column !== 'phone') {
            return $value;
        }

        if (!is_array($item)) {
            return $value;
        }

        $attendee_id = isset($item['attendee_id']) ? $item['attendee_id'] : 0;
        $phone = self::get_attendee_phone($attendee_id);

        return $phone ? esc_html($phone) : '—';
    }

    /**
     * Add phone to searchable fields
     */
    public function add_phone_search($keys) {
        if (!is_array($keys)) {
            return $keys;
        }
        $keys[] = self::PHONE_META_KEY;
        return $keys;
    }

    /**
     * Add phone to REST API response
     */
    public function add_phone_to_api($data, $attendee_id) {
        if (!is_array($data)) {
            return $data;
        }
        $data['phone'] = self::get_attendee_phone($attendee_id);
        return $data;
    }

    /**
     * Get all attendees with phone for an event
     */
    public static function get_event_attendees_with_phones($event_id) {
        $attendees = array();

        // Try using Tribe's function
        if (function_exists('tribe_tickets_get_attendees')) {
            $all_attendees = tribe_tickets_get_attendees($event_id);

            if (!is_array($all_attendees)) {
                return $attendees;
            }

            foreach ($all_attendees as $attendee) {
                if (!is_array($attendee)) {
                    continue;
                }

                $attendee_id = isset($attendee['attendee_id']) ? $attendee['attendee_id'] : 0;
                $phone = self::get_attendee_phone($attendee_id);

                if (!empty($phone)) {
                    $attendees[] = array(
                        'id'    => $attendee_id,
                        'name'  => isset($attendee['holder_name']) ? $attendee['holder_name'] : '',
                        'email' => isset($attendee['holder_email']) ? $attendee['holder_email'] : '',
                        'phone' => $phone,
                        'ticket_name' => isset($attendee['ticket']) ? $attendee['ticket'] : '',
                    );
                }
            }
        }

        return $attendees;
    }
}
