<?php
/**
 * SMSTunnel Events Calendar - Per-Event Settings
 *
 * Adds SMS settings metabox to event edit screen
 */

if (!defined('ABSPATH')) {
    exit;
}

class SMSTunnel_TEC_Event_Settings {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('save_post_tribe_events', array($this, 'save_meta_box'), 10, 2);

        // Add column to events list
        add_filter('manage_tribe_events_posts_columns', array($this, 'add_sms_column'));
        add_action('manage_tribe_events_posts_custom_column', array($this, 'render_sms_column'), 10, 2);

        // AJAX for manual reminder
        add_action('wp_ajax_smstunnel_send_manual_reminder', array($this, 'ajax_send_reminder'));
    }

    /**
     * Add meta box
     */
    public function add_meta_box() {
        add_meta_box(
            'smstunnel_event_sms',
            __('SMS Notifications', 'smstunnel-events-calendar'),
            array($this, 'render_meta_box'),
            'tribe_events',
            'side',
            'default'
        );
    }

    /**
     * Render meta box
     */
    public function render_meta_box($post) {
        wp_nonce_field('smstunnel_event_sms', 'smstunnel_event_sms_nonce');

        $sms_enabled = get_post_meta($post->ID, '_smstunnel_sms_enabled', true);
        $reminder_enabled = get_post_meta($post->ID, '_smstunnel_reminder_enabled', true);
        $organizer_phone = get_post_meta($post->ID, '_smstunnel_organizer_phone', true);

        // Default to enabled
        if ($sms_enabled === '') {
            $sms_enabled = '1';
        }
        if ($reminder_enabled === '') {
            $reminder_enabled = '1';
        }

        // Get reminder status
        $reminder_status = SMSTunnel_TEC_Reminders::get_reminder_status($post->ID);

        // Get attendee count
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($post->ID);
        $attendee_count = count($attendees);
        ?>
        <div class="smstunnel-event-settings">
            <p>
                <label>
                    <input type="checkbox" name="smstunnel_sms_enabled" value="1" <?php checked($sms_enabled, '1'); ?>>
                    <?php esc_html_e('Enable SMS for this event', 'smstunnel-events-calendar'); ?>
                </label>
            </p>

            <p>
                <label>
                    <input type="checkbox" name="smstunnel_reminder_enabled" value="1" <?php checked($reminder_enabled, '1'); ?>>
                    <?php esc_html_e('Enable reminder SMS', 'smstunnel-events-calendar'); ?>
                </label>
            </p>

            <p>
                <label for="smstunnel_organizer_phone">
                    <?php esc_html_e('Organizer Phone:', 'smstunnel-events-calendar'); ?>
                </label>
                <input type="tel" id="smstunnel_organizer_phone" name="smstunnel_organizer_phone"
                       value="<?php echo esc_attr($organizer_phone); ?>"
                       placeholder="+40712345678"
                       class="widefat">
                <span class="description"><?php esc_html_e('Override default organizer phone for this event.', 'smstunnel-events-calendar'); ?></span>
            </p>

            <hr>

            <p class="smstunnel-stats">
                <strong><?php esc_html_e('SMS Stats:', 'smstunnel-events-calendar'); ?></strong><br>
                <?php
                /* translators: %d: number of attendees with phone numbers */
                printf(esc_html__('Attendees with phone: %d', 'smstunnel-events-calendar'), absint($attendee_count));
                ?>
            </p>

            <?php if ($reminder_status['sent']): ?>
            <p class="smstunnel-reminder-status sent">
                <span class="dashicons dashicons-yes-alt"></span>
                <?php
                printf(
                    /* translators: %1$d: number of reminders sent, %2$s: date and time when sent */
                    esc_html__('Reminders sent: %1$d on %2$s', 'smstunnel-events-calendar'),
                    absint($reminder_status['count']),
                    esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($reminder_status['sent_at'])))
                );
                ?>
            </p>
            <?php else: ?>
            <p class="smstunnel-reminder-status pending">
                <span class="dashicons dashicons-clock"></span>
                <?php esc_html_e('Reminders not yet sent', 'smstunnel-events-calendar'); ?>
            </p>
            <?php endif; ?>

            <?php if ($attendee_count > 0): ?>
            <p>
                <button type="button" class="button smstunnel-send-reminder" data-event-id="<?php echo esc_attr($post->ID); ?>">
                    <span class="dashicons dashicons-email-alt"></span>
                    <?php esc_html_e('Send Reminder Now', 'smstunnel-events-calendar'); ?>
                </button>
            </p>
            <?php endif; ?>
        </div>

        <style>
            .smstunnel-event-settings p { margin: 10px 0; }
            .smstunnel-event-settings .description { font-size: 11px; color: #666; display: block; margin-top: 4px; }
            .smstunnel-reminder-status { padding: 8px; border-radius: 4px; }
            .smstunnel-reminder-status.sent { background: #d4edda; color: #155724; }
            .smstunnel-reminder-status.pending { background: #fff3cd; color: #856404; }
            .smstunnel-reminder-status .dashicons { vertical-align: middle; margin-right: 4px; }
            .smstunnel-send-reminder .dashicons { vertical-align: middle; margin-right: 4px; font-size: 16px; }
        </style>

        <script>
        jQuery(document).ready(function($) {
            $('.smstunnel-send-reminder').on('click', function(e) {
                e.preventDefault();
                var $btn = $(this);
                var eventId = $btn.data('event-id');

                if (!confirm('<?php echo esc_js(__('Send reminder SMS to all attendees now?', 'smstunnel-events-calendar')); ?>')) {
                    return;
                }

                $btn.prop('disabled', true).text('<?php echo esc_js(__('Sending...', 'smstunnel-events-calendar')); ?>');

                $.post(ajaxurl, {
                    action: 'smstunnel_send_manual_reminder',
                    event_id: eventId,
                    nonce: '<?php echo esc_attr(wp_create_nonce('smstunnel_manual_reminder')); ?>'
                }, function(response) {
                    if (response.success) {
                        alert('<?php echo esc_js(__('Reminders sent!', 'smstunnel-events-calendar')); ?> ' + response.data.count + ' SMS');
                        location.reload();
                    } else {
                        alert('<?php echo esc_js(__('Error:', 'smstunnel-events-calendar')); ?> ' + response.data);
                    }
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-email-alt"></span> <?php echo esc_js(__('Send Reminder Now', 'smstunnel-events-calendar')); ?>');
                });
            });
        });
        </script>
        <?php
    }

    /**
     * Save meta box
     */
    public function save_meta_box($post_id, $post) {
        // Check nonce
        if (!isset($_POST['smstunnel_event_sms_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['smstunnel_event_sms_nonce'])), 'smstunnel_event_sms')) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Don't save on autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Save settings
        update_post_meta($post_id, '_smstunnel_sms_enabled', isset($_POST['smstunnel_sms_enabled']) ? '1' : '0');
        update_post_meta($post_id, '_smstunnel_reminder_enabled', isset($_POST['smstunnel_reminder_enabled']) ? '1' : '0');

        if (isset($_POST['smstunnel_organizer_phone'])) {
            update_post_meta($post_id, '_smstunnel_organizer_phone', sanitize_text_field(wp_unslash($_POST['smstunnel_organizer_phone'])));
        }

        // Check if event date changed - reset reminder status
        $old_start = get_post_meta($post_id, '_EventStartDate', true);
        if (isset($_POST['EventStartDate']) && sanitize_text_field(wp_unslash($_POST['EventStartDate'])) !== $old_start) {
            SMSTunnel_TEC_Reminders::reset_reminder_status($post_id);
        }
    }

    /**
     * Add SMS column to events list
     */
    public function add_sms_column($columns) {
        $columns['smstunnel_sms'] = __('SMS', 'smstunnel-events-calendar');
        return $columns;
    }

    /**
     * Render SMS column
     */
    public function render_sms_column($column, $post_id) {
        if ($column !== 'smstunnel_sms') {
            return;
        }

        $sms_enabled = get_post_meta($post_id, '_smstunnel_sms_enabled', true);
        $attendees = SMSTunnel_TEC_Phone_Field::get_event_attendees_with_phones($post_id);
        $count = count($attendees);

        if ($sms_enabled === '0') {
            echo '<span class="dashicons dashicons-no-alt" style="color: #999;" title="' . esc_attr__('SMS disabled', 'smstunnel-events-calendar') . '"></span>';
        } else {
            echo '<span class="dashicons dashicons-smartphone" style="color: #0073aa;" title="' . esc_attr__('SMS enabled', 'smstunnel-events-calendar') . '"></span>';
            echo ' <small>(' . absint($count) . ')</small>';
        }
    }

    /**
     * AJAX handler for manual reminder
     */
    public function ajax_send_reminder() {
        check_ajax_referer('smstunnel_manual_reminder', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('Permission denied', 'smstunnel-events-calendar'));
        }

        $event_id = isset($_POST['event_id']) ? absint($_POST['event_id']) : 0;
        if (!$event_id) {
            wp_send_json_error(__('Invalid event', 'smstunnel-events-calendar'));
        }

        $status = SMSTunnel_TEC_Reminders::send_manual_reminder($event_id);

        wp_send_json_success(array(
            'count' => $status['count'],
        ));
    }
}
