<?php
/**
 * Plugin Name: SMSTunnel Elementor Forms
 * Plugin URI: https://smstunnel.io/integrations/elementor-forms
 * Description: Send SMS notifications via SMSTunnel when Elementor Forms are submitted. Adds an SMS action to Elementor Form widget.
 * Version: 1.0.0
 * Author: SMSTunnel.io
 * Author URI: https://smstunnel.io
 * Text Domain: smstunnel-elementor-forms
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Requires Plugins: elementor-pro
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @author Narcis Bodea <contact@narboweb.ro>
 * @copyright 2024-2025 NARBOWEB SRL
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SMSTUNNEL_ELEMENTOR_VERSION', '1.0.0');
define('SMSTUNNEL_ELEMENTOR_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMSTUNNEL_ELEMENTOR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SMSTUNNEL_ELEMENTOR_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main SMSTunnel Elementor Forms Plugin Class
 */
class SMSTunnel_Elementor_Forms {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        add_action('admin_notices', array($this, 'check_dependencies'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check dependencies
        if (!$this->dependencies_met()) {
            return;
        }

        // Register form action
        add_action('elementor_pro/forms/actions/register', array($this, 'register_form_action'));

        // Admin assets
        add_action('admin_enqueue_scripts', array($this, 'admin_assets'));
    }

    /**
     * Check if dependencies are met
     */
    private function dependencies_met() {
        return did_action('elementor_pro/init');
    }

    /**
     * Display admin notices for missing dependencies
     */
    public function check_dependencies() {
        if (!did_action('elementor_pro/init')) {
            echo '<div class="notice notice-error"><p>';
            echo wp_kses(
                __('<strong>SMSTunnel Elementor Forms</strong> requires <strong>Elementor Pro</strong> to be installed and activated.', 'smstunnel-elementor-forms'),
                array('strong' => array())
            );
            echo '</p></div>';
            return;
        }

        if (!function_exists('smstunnel_send_sms')) {
            echo '<div class="notice notice-warning is-dismissible"><p>';
            echo wp_kses(
                sprintf(
                    /* translators: %s: URL to SMSTunnel plugin page */
                    __('<strong>SMSTunnel Elementor Forms</strong>: The SMSTunnel plugin is not active. SMS notifications will not be sent until you install and configure the <a href="%s" target="_blank">SMSTunnel</a> plugin.', 'smstunnel-elementor-forms'),
                    'https://smstunnel.io/wordpress'
                ),
                array(
                    'strong' => array(),
                    'a'      => array(
                        'href'   => array(),
                        'target' => array(),
                    ),
                )
            );
            echo '</p></div>';
        }
    }

    /**
     * Register form action
     */
    public function register_form_action($form_actions_registrar) {
        require_once SMSTUNNEL_ELEMENTOR_PLUGIN_DIR . 'includes/class-sms-action.php';

        $form_actions_registrar->register(new SMSTunnel_Elementor_SMS_Action());
        $form_actions_registrar->register(new SMSTunnel_Elementor_SMS_2_Action());
    }

    /**
     * Enqueue admin assets
     */
    public function admin_assets($hook) {
        wp_enqueue_style(
            'smstunnel-elementor-admin',
            SMSTUNNEL_ELEMENTOR_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SMSTUNNEL_ELEMENTOR_VERSION
        );
    }
}

// Initialize plugin
add_action('plugins_loaded', function() {
    SMSTunnel_Elementor_Forms::get_instance();
}, 20);
