<?php
/**
 * SMSTunnel Elementor Forms - SMS Action Classes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SMS Action for Elementor Forms
 */
class SMSTunnel_Elementor_SMS_Action extends \ElementorPro\Modules\Forms\Classes\Action_Base {

    /**
     * Get action name
     */
    public function get_name() {
        return 'smstunnel_sms';
    }

    /**
     * Get action label
     */
    public function get_label() {
        return __('SMSTunnel SMS', 'smstunnel-elementor-forms');
    }

    /**
     * Register action controls (settings)
     */
    public function register_settings_section($widget) {
        $widget->start_controls_section(
            'section_smstunnel_sms',
            array(
                'label'     => __('SMSTunnel SMS', 'smstunnel-elementor-forms'),
                'condition' => array(
                    'submit_actions' => $this->get_name(),
                ),
            )
        );

        $widget->add_control(
            'smstunnel_sms_recipient',
            array(
                'label'       => __('Recipient Phone Number(s)', 'smstunnel-elementor-forms'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'placeholder' => '+40712345678',
                'description' => __('Enter phone number(s) in international format. Separate multiple numbers with commas. Use [field id="phone"] to get value from form field.', 'smstunnel-elementor-forms'),
                'label_block' => true,
            )
        );

        $widget->add_control(
            'smstunnel_sms_body',
            array(
                'label'       => __('Message Body', 'smstunnel-elementor-forms'),
                'type'        => \Elementor\Controls_Manager::TEXTAREA,
                'default'     => __("New form submission:\n[all-fields]", 'smstunnel-elementor-forms'),
                'description' => __('Use [field id="field_id"] to include form data. Use [all-fields] for all fields. Maximum 1600 characters.', 'smstunnel-elementor-forms'),
                'rows'        => 6,
            )
        );

        $widget->add_control(
            'smstunnel_sms_shortcodes_help',
            array(
                'type'            => \Elementor\Controls_Manager::RAW_HTML,
                'raw'             => '<div style="padding: 10px; background: #f9f9f9; border-radius: 4px;">
                    <strong>' . __('Available shortcodes:', 'smstunnel-elementor-forms') . '</strong><br>
                    <code>[field id="field_id"]</code> - ' . __('Field value', 'smstunnel-elementor-forms') . '<br>
                    <code>[all-fields]</code> - ' . __('All submitted fields', 'smstunnel-elementor-forms') . '
                </div>',
                'content_classes' => 'elementor-descriptor',
            )
        );

        $widget->end_controls_section();
    }

    /**
     * Run action on form submission
     */
    public function run($record, $ajax_handler) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $settings = $record->get('form_settings');

        $recipient = isset($settings['smstunnel_sms_recipient']) ? $settings['smstunnel_sms_recipient'] : '';
        $body = isset($settings['smstunnel_sms_body']) ? $settings['smstunnel_sms_body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return;
        }

        // Get form fields
        $raw_fields = $record->get('fields');
        $fields = array();
        foreach ($raw_fields as $id => $field) {
            $fields[$id] = $field['value'];
        }

        // Replace shortcodes
        $recipient = $this->replace_shortcodes($recipient, $fields, $raw_fields);
        $body = $this->replace_shortcodes($body, $fields, $raw_fields);

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Parse recipients (comma-separated)
        $recipients = array_filter(array_map('trim', explode(',', $recipient)));

        // Send SMS to each recipient
        foreach ($recipients as $phone) {
            if (!empty($phone)) {
                smstunnel_send_sms($phone, $body, array(
                    'source' => 'elementor-forms'
                ));
            }
        }
    }

    /**
     * Replace shortcodes in text
     */
    private function replace_shortcodes($text, $fields, $raw_fields) {
        // Replace [field id="X"]
        if (preg_match_all('/\[field\s+id=["\']?([^"\']+)["\']?\]/i', $text, $matches)) {
            foreach ($matches[1] as $index => $field_id) {
                $value = isset($fields[$field_id]) ? $fields[$field_id] : '';
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace [all-fields]
        if (strpos($text, '[all-fields]') !== false) {
            $all_fields = '';
            foreach ($raw_fields as $id => $field) {
                if (!empty($field['value'])) {
                    $label = !empty($field['title']) ? $field['title'] : $id;
                    $all_fields .= $label . ': ' . $field['value'] . "\n";
                }
            }
            $text = str_replace('[all-fields]', trim($all_fields), $text);
        }

        return $text;
    }

    /**
     * On export
     */
    public function on_export($element) {
        unset($element['smstunnel_sms_recipient']);
        unset($element['smstunnel_sms_body']);
        return $element;
    }
}

/**
 * SMS (2) Autoresponder Action for Elementor Forms
 */
class SMSTunnel_Elementor_SMS_2_Action extends \ElementorPro\Modules\Forms\Classes\Action_Base {

    /**
     * Get action name
     */
    public function get_name() {
        return 'smstunnel_sms_2';
    }

    /**
     * Get action label
     */
    public function get_label() {
        return __('SMSTunnel SMS (2) Autoresponder', 'smstunnel-elementor-forms');
    }

    /**
     * Register action controls (settings)
     */
    public function register_settings_section($widget) {
        $widget->start_controls_section(
            'section_smstunnel_sms_2',
            array(
                'label'     => __('SMSTunnel SMS (2) Autoresponder', 'smstunnel-elementor-forms'),
                'condition' => array(
                    'submit_actions' => $this->get_name(),
                ),
            )
        );

        $widget->add_control(
            'smstunnel_sms_2_description',
            array(
                'type'            => \Elementor\Controls_Manager::RAW_HTML,
                'raw'             => '<p>' . __('SMS (2) is used to send a confirmation message to the form submitter.', 'smstunnel-elementor-forms') . '</p>',
                'content_classes' => 'elementor-descriptor',
            )
        );

        $widget->add_control(
            'smstunnel_sms_2_recipient',
            array(
                'label'       => __('Recipient Phone Number', 'smstunnel-elementor-forms'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'placeholder' => '[field id="phone"]',
                'description' => __('Use [field id="phone"] to get the phone number from a form field.', 'smstunnel-elementor-forms'),
                'label_block' => true,
            )
        );

        $widget->add_control(
            'smstunnel_sms_2_body',
            array(
                'label'       => __('Confirmation Message', 'smstunnel-elementor-forms'),
                'type'        => \Elementor\Controls_Manager::TEXTAREA,
                'default'     => __("Thank you for contacting us!\nWe have received your message and will get back to you soon.", 'smstunnel-elementor-forms'),
                'description' => __('Confirmation message sent to the form submitter. Maximum 1600 characters.', 'smstunnel-elementor-forms'),
                'rows'        => 5,
            )
        );

        $widget->end_controls_section();
    }

    /**
     * Run action on form submission
     */
    public function run($record, $ajax_handler) {
        // Check if SMSTunnel is available
        if (!function_exists('smstunnel_send_sms')) {
            return;
        }

        $settings = $record->get('form_settings');

        $recipient = isset($settings['smstunnel_sms_2_recipient']) ? $settings['smstunnel_sms_2_recipient'] : '';
        $body = isset($settings['smstunnel_sms_2_body']) ? $settings['smstunnel_sms_2_body'] : '';

        // Check if we have recipient and body
        if (empty($recipient) || empty($body)) {
            return;
        }

        // Get form fields
        $raw_fields = $record->get('fields');
        $fields = array();
        foreach ($raw_fields as $id => $field) {
            $fields[$id] = $field['value'];
        }

        // Replace shortcodes
        $recipient = $this->replace_shortcodes($recipient, $fields, $raw_fields);
        $body = $this->replace_shortcodes($body, $fields, $raw_fields);

        // Skip if recipient is empty after replacement
        if (empty(trim($recipient))) {
            return;
        }

        // Trim body to max 1600 chars
        $body = mb_substr($body, 0, 1600);

        // Send SMS
        smstunnel_send_sms($recipient, $body, array(
            'source' => 'elementor-forms-autoresponder'
        ));
    }

    /**
     * Replace shortcodes in text
     */
    private function replace_shortcodes($text, $fields, $raw_fields) {
        // Replace [field id="X"]
        if (preg_match_all('/\[field\s+id=["\']?([^"\']+)["\']?\]/i', $text, $matches)) {
            foreach ($matches[1] as $index => $field_id) {
                $value = isset($fields[$field_id]) ? $fields[$field_id] : '';
                $text = str_replace($matches[0][$index], $value, $text);
            }
        }

        // Replace [all-fields]
        if (strpos($text, '[all-fields]') !== false) {
            $all_fields = '';
            foreach ($raw_fields as $id => $field) {
                if (!empty($field['value'])) {
                    $label = !empty($field['title']) ? $field['title'] : $id;
                    $all_fields .= $label . ': ' . $field['value'] . "\n";
                }
            }
            $text = str_replace('[all-fields]', trim($all_fields), $text);
        }

        return $text;
    }

    /**
     * On export
     */
    public function on_export($element) {
        unset($element['smstunnel_sms_2_recipient']);
        unset($element['smstunnel_sms_2_body']);
        return $element;
    }
}
